<?php

namespace block_ned_student_menu;
use \block_ned_student_menu\shared_lib as SH;

/**
 * Used to save some global values as one global array
 * @param array $keys - path to the value (example: ['category1', 'category2', 'category3']
 * @param null  $value - saved by $keys if provided
 * @param bool  $set_even_null - set in true, if you need to save $value === null
 *
 * @return array|mixed|null
 */
function ned_global($keys=[], $value=null, $set_even_null=false){
    global $ned_globals_list;
    $ned_globals_list = isset($ned_globals_list) ? $ned_globals_list : [];
    $obj = &$ned_globals_list;
    $set_val = !is_null($value) || $set_even_null;
    foreach ($keys as $key){
        if (!isset($obj[$key])){
            if (!$set_val){
                return null;
            }
            $obj[$key] = [];
        }
        $obj = &$obj[$key];
    }
    if ($set_val){
        $obj = $value;
    }
    return $obj;
}

function _DB(){
    global $DB;
    return $DB;
}

/**
 * @param $courseid - course id ($mod->course)
 * @param $itemmodule - module name (component name without 'mod_') ($mod->itemmodule)
 * @param $iteminstance - instance id ($mod->instance)
 *
 * @return \stdClass|bool Returns a {$itemmodule} instance or false if none found
 * @throws \dml_exception
 */
function get_module_db($courseid, $itemmodule, $iteminstance){
    $G_NAME = 'modules';
    $modules_list = ned_global([$G_NAME, $courseid]);
    if(is_null($modules_list)){
        $modules_list = [];
        $availabletypes = \format_singleactivity::get_supported_activities();
        foreach ($availabletypes as $module => $name){
            $modules_list[$module] = [];
        }
        ned_global([$G_NAME, $courseid], $modules_list);
    }
    if(isset($modules_list[$itemmodule]) and $modules_list[$itemmodule] !== false){
        if(empty($modules_list[$itemmodule])){
            $mods = _DB()->get_records($itemmodule, ['course' => $courseid]);
            if(!empty($mods)){
                foreach ($mods as $mod){
                    $modules_list[$itemmodule][$mod->id] = $mod;
                }
            } else {
                $modules_list[$itemmodule] = false;
            }
            ned_global([$G_NAME, $courseid], $modules_list);
        }
    }

    if(isset($modules_list[$itemmodule][$iteminstance])){
        return $modules_list[$itemmodule][$iteminstance];
    } else {
        return false;
    }
}

/**
 * @param $userid - user id
 * @param $itemid - grade_item id ($grade_item->id)
 *
 * @return \stdClass|bool Returns a grade_grade instance or false if none found
 * @throws \dml_exception
 */
function get_grade_grade($userid, $itemid) {
    $G_NAME = 'grade_grades';
    $grades_list = ned_global([$G_NAME, $userid]);
    if (is_null($grades_list)){
        $grades_list= [];
        $grades = _DB()->get_records('grade_grades', ['userid' => $userid]);
        if (!empty($grades)){
            foreach ($grades as $grade){
                $grades_list[$grade->itemid] = $grade;
            }
        }
        ned_global([$G_NAME, $userid], $grades_list);
    }

    if (isset($grades_list[$itemid])){
        return $grades_list[$itemid];
    } else {
        return false;
    }
}

/**
 * @param $courseid - course id ($mod->course)
 * @param $itemmodule - module name (component name without 'mod_') ($mod->modname)
 * @param $iteminstance - instance id ($mod->instance)
 *
 * @return \grade_item|bool Returns a grade_item instance or false if none found
 */
function get_grade_item($courseid, $itemmodule, $iteminstance){
    $G_NAME = 'grade_items';
    $grades_list = ned_global([$G_NAME, $courseid]);
    if (is_null($grades_list)){
        $grades_list = [];
        $grade_items = \grade_item::fetch_all(['itemtype' => 'mod', 'courseid' => $courseid]);
        if (!empty($grade_items)){
            foreach ($grade_items as $grade_item){
                $grades_list[$grade_item->itemmodule][$grade_item->iteminstance] = $grade_item;
            }
        }
        ned_global([$G_NAME, $courseid], $grades_list);
    }

    if (isset($grades_list[$itemmodule][$iteminstance])){
        return $grades_list[$itemmodule][$iteminstance];
    } else {
        return false;
    }
}

/**
 * @param \stdClass $course
 * @param bool      $for_deadline_manager
 *
 * @return array - $allowedgroups (array), $ignore_grouping (bool)
 */
function get_groups($course, $for_deadline_manager=false){
    global $USER;
    $G_NAME = 'groups';
    $userid = $USER->id;
    $courseid = $course->id;
    $for_deadline_manager = (int)$for_deadline_manager;
    $data =  ned_global([$G_NAME, $courseid, $userid, $for_deadline_manager]);

    if (is_null($data)){
        $groupmode = $course->groupmode;
        $context = \context_course::instance($courseid);
        $aag = $for_deadline_manager ?  has_capability('block/ned_teacher_tools:view_all_groups', $context) :
            has_capability('moodle/site:accessallgroups', $context);

        if (!$groupmode){
            $allowedgroups = [SH::stdClass2(['id' => 0, 'name' => str('allusers')])];
        } elseif ($groupmode == VISIBLEGROUPS or $aag) {
            $allowedgroups = groups_get_all_groups($courseid, 0, $course->defaultgroupingid);
        } else {
            $allowedgroups = groups_get_all_groups($courseid, $userid, $course->defaultgroupingid);
            if ($for_deadline_manager && \block_ned_teacher_tools\deadline_manager::can_view_own_groups($context)) {
                foreach ($allowedgroups as $key => $allowedgroup) {
                    if ($allowedgroup->schedule != \block_ned_teacher_tools\deadline_manager::SCHEDULE_FULL || !groups_is_member($allowedgroup->id)) {
                        unset($allowedgroups[$key]);
                    }
                }
            }
        }

        $ignore_grouping = !$groupmode || $aag;
        $data = [$allowedgroups, $ignore_grouping];
        ned_global([$G_NAME, $courseid, $userid, $for_deadline_manager], $data);
    }

    // list($allowedgroups, $ignore_grouping) = $data;
    return $data;
}

/**
 * @param \stdClass $course
 * @param int       $teacherid
 * @param int       $groupid
 * @param null      $show_only_active
 *
 * @return array of students ['id' => stdClass] by teacher $userid. If it's student, return array of it's record
 */
function get_students(\stdClass $course, $teacherid=0, $groupid=0, $show_only_active=null){
    global $USER;
    $teacherid = $teacherid ? $teacherid : $USER->id;
    $courseid = $course->id;
    $G_NAME = 'students';
    $student_list = ned_global([$G_NAME, $courseid, $teacherid, $groupid]);

    if (is_null($student_list)){
        $student_list = [];
        $context = \context_course::instance($course->id, MUST_EXIST);
        $isteacher = has_capability('moodle/grade:viewall', $context, $teacherid);
        if (!$isteacher){
            $student_list[$teacherid] = \core_user::get_user($teacherid);
        } else {
            $show_only_active = $show_only_active ?? !get_cache_show_inactive($courseid);
            $student_list = get_course_students_by_role($courseid, $groupid, $show_only_active);
        }
        ned_global([$G_NAME, $courseid, $teacherid, $groupid], $student_list);
    }

    return $student_list;
}

/**
 * Return list of students ['id' => stdClass] by $courseid & $groupid, or false if there no such students.
 *
 * @param int  $courseid
 * @param int  $groupid
 * @param null $show_only_active
 * @param null $load_cm_students
 *
 * @return false|array
 */
function get_course_students_by_role($courseid, $groupid=0, $show_only_active=null, $load_cm_students=null){
    global $DB;
    $G_NAME = 'role_students';
    $show_only_active = $show_only_active ?? !get_cache_show_inactive($courseid);
    $show_only_active = (int)$show_only_active;
    $load_cm_students = (int)$load_cm_students;
    $student_list = ned_global([$G_NAME, $courseid, $groupid, $show_only_active, $load_cm_students]);

    if (is_null($student_list)){
        $params = [];
        $where = [];
        if ($load_cm_students){
            $where[] = "r.id = role_cm.id";
        } else {
            $where[] = "r.id = role_course.id";
        }
        $sql = get_sql_user_enrolments(
            "DISTINCT u.id, u.*, gr.id AS groupid, role_course.shortname AS course_role",
            $where, $params, ROLE_STUDENT,
            $courseid,
            $groupid, 0, $show_only_active, 'GROUP BY u.id ORDER BY u.firstname, u.lastname');
        $student_list = $DB->get_records_sql($sql, $params);

        ned_global([$G_NAME, $courseid, $groupid, $show_only_active, $load_cm_students], $student_list);
    }

    return $student_list;
}

/**
 * @param \stdClass $course
 * @param bool      $for_deadline_manager
 * @param null      $show_only_active
 * @param null      $load_cm_students
 *
 * @return array - $users (array), $allowedgroups(array)
 */
function get_users_and_groups($course, $for_deadline_manager=false, $show_only_active=null, $load_cm_students=null) {
    global $USER;
    $G_NAME = 'users_and_groups';
    $userid = $USER->id;
    $courseid = $course->id;
    $for_deadline_manager = (int)$for_deadline_manager;
    $show_only_active = is_null($show_only_active) ? !get_cache_show_inactive($course->id) : $show_only_active;
    $show_only_active = (int)$show_only_active;
    $load_cm_students = (int)$load_cm_students;
    $data = ned_global([$G_NAME, $courseid, $userid, $for_deadline_manager, $show_only_active, $load_cm_students]);

    if (is_null($data)){
        list($allowedgroups, $ignore_grouping) = get_groups($course, $for_deadline_manager);

        $users = [];
        if ($ignore_grouping) {
            $users = get_course_students_by_role($course->id, 0, $show_only_active, $load_cm_students);
            if (count($allowedgroups) > 1 || !isset($allowedgroups[0])){
                foreach ($allowedgroups as $key => $group){
                    $allowedgroups[$key]->users = [];
                }
                foreach ($users as $user_key => $user){
                    $groupid = $user->groupid ?? 0;
                    if (isset($allowedgroups[$groupid])){
                        $user->group = $allowedgroups[$groupid];
                        $allowedgroups[$groupid]->users[$user_key] = $user;
                    } else {
                        $user->group = null;
                    }
                }
                    }
                } else {
                    foreach ($allowedgroups as $key => $group){
                $enrolledusers = get_course_students_by_role($course->id, $group->id, $show_only_active, $load_cm_students);

                foreach ($enrolledusers as $user_key => $user){
                    $user->group = $group;
                    }

                $allowedgroups[$key]->users = $enrolledusers;
                $users += $enrolledusers;
                }
            }

        $data = [$users, $allowedgroups];
        ned_global([$G_NAME, $courseid, $userid, $for_deadline_manager, $show_only_active, $load_cm_students], $data);
    }

    // list($users, $allowedgroups) = $data;
    return $data;
}

/**
 * @param $modid - mod id
 *
 * @return array Returns array of tags, maybe empty
 */
function get_tags($modid) {
    $G_NAME = 'tags';
    $tag_list = ned_global([$G_NAME, $modid]);

    if (is_null($tag_list)){
        $tag_list = \core_tag_tag::get_item_tags_array('core', 'course_modules', $modid);
        if (empty($tag_list)){
            $tag_list = [];
        } else {
            $all_tags = get_all_tags();
            if (is_null($all_tags)){
                $all_tags = [];
            }
            $all_tags += $tag_list;
            ned_global([$G_NAME, 0], $all_tags);
            ned_global([$G_NAME, $modid], $tag_list);
        }
    }

    return $tag_list;
}


/**
 * Returns array of all checked tags, maybe empty
 * Note: it's not array of all tags from somewhere, it's array of all tags which were get by get_tags functions
 *
 * @return array
 */
function get_all_tags() {
    $G_NAME = 'tags';
    $tag_list = ned_global([$G_NAME, 0]);
    if (is_null($tag_list)){
        $tag_list = [];
    }

    return $tag_list;
}

/**
 * Returns instance of mod (activity)
 * @param $modname
 * @param $instanceid
 *
 * @return array|mixed|null
 * @throws \dml_exception
 */
function get_instance($modname, $instanceid){
    $G_NAME = 'instances';
    $inst = ned_global([$G_NAME, $modname, $instanceid]);
    if (is_null($inst)){
        $inst = _DB()->get_record($modname, ['id' => $instanceid]);
        ned_global([$G_NAME, $modname, $instanceid], $inst);
    }
    return $inst;
}

/**
 * Return id of kica_item, 0 if item not founded
 * @param $courseid
 * @param $itemmodule
 * @param $iteminstance
 *
 * @return null|int
 */
function get_kica_item_id($courseid, $itemmodule, $iteminstance){
    if (!is_kica_exists()){
        return null;
    }
    $G_NAME = 'kica_item_id';
    $kica_item_id = ned_global([$G_NAME, $courseid, $itemmodule, $iteminstance]);
    if (is_null($kica_item_id)){
        $gradeitem = get_grade_item($courseid, $itemmodule, $iteminstance);
        if ($gradeitem){
            $kica = _DB()->get_record('local_kica_grade_items', array('itemid' => $gradeitem->id));
            $kica_item_id = $kica ? $kica->id : 0;
        } else {
            $kica_item_id = 0;
        }
        ned_global([$G_NAME, $courseid, $itemmodule, $iteminstance], $kica_item_id);
    }
    return $kica_item_id;
}

/**
 * Return result the same to is_graded() from kica class
 * @param $userid
 * @param $kicaitemid
 *
 * @return array|bool|mixed|null
 */
function get_kica_is_graded($userid, $kicaitemid){
    if (!is_kica_exists()){
        return null;
    }
    $G_NAME = 'kica_is_graded';
    $kica_is_graded = ned_global([$G_NAME, $userid, $kicaitemid]);
    if (is_null($kica_is_graded)){
        if (ned_global(['load_kica_grade_item_id_and_graded_for_all_users'])){
            return false;
        }
        if ($grade = _DB()->get_record('local_kica_grade_grades', ['userid' => $userid, 'itemid' => $kicaitemid])) {
            $kica_is_graded = (!is_null($grade->knowledge) || !is_null($grade->inquiry) || !is_null($grade->communication) || !is_null
                ($grade->application));
        } else {
            $kica_is_graded = false;
        }
        ned_global([$G_NAME, $userid, $kicaitemid], $kica_is_graded);
    }

    return $kica_is_graded;
}

// Prepare big amounts of data


/**
 * Load data from grade_items & grade_grades for course (and specific user, if set user id)
 *
 * @param      $courseid
 * @param null $userid
 *
 * @throws \dml_exception
 */
function load_mod_grade_items($courseid, $userid=null){
    if (ned_global(['load_mod_grade_items'])){
        return;
    }

    list($gg_l, $gi_l, $sc_l) = array([], [], []);
    $t = ['grade_grades' => 'g_g', 'grade_items' => 'g_i'];
    $t_cols = [];
    $dl = '___';
    $fields = [];
    foreach ($t as $t_name => $alias){
        $cols = _DB()->get_columns($t_name);
        foreach ($cols as $col => $col_info){
            $t_cols[$t_name][$col] = $alias.$dl.$col;
            $fields[] = "$alias.$col AS {$t_cols[$t_name][$col]}";
        }
        $t[$t_name] = $alias.$dl;
    }

    $sql = 'SELECT CONCAT(g_i.id, g_g.id), ' . join(',', $fields) . '
                FROM 
                  {grade_items} g_i
                LEFT JOIN {grade_grades} g_g
                  ON g_g.itemid = g_i.id
                WHERE g_i.courseid = :courseid AND g_i.itemtype = "mod" AND g_g.id
        ';
    $params = ['courseid' => $courseid];
    if (!is_null($userid)){
        $sql .= ' AND g_g.userid = :userid';
        $params['userid'] = $userid;
    }

    $grade_info_records = _DB()->get_records_sql($sql, $params);
    if (empty($grade_info_records)){
        return;
    }

    $gi_list = [];
    $gg_list = [];
    foreach ($grade_info_records as $r){
        $gi = new \stdClass();
        foreach ($t_cols['grade_items'] as $col => $alias_col){
            $gi->$col = $r->$alias_col;
        }
        $gi_list[$gi->itemmodule][$gi->iteminstance] = $gi;

        $gg = new \stdClass();
        foreach ($t_cols['grade_grades'] as $col => $alias_col){
            $gg->$col = $r->$alias_col;
        }
        $gg_list[$gg->userid][$gg->itemid] = $gg;
    }

    ned_global(['grade_items', $courseid], $gi_list);
    ned_global(['grade_grades'], $gg_list);
    ned_global(['load_mod_grade_items'], 1);
}

/**
 * Load data from local_kica_grade_items & local_kica_grade_grades for course (and specific user, if set user id)
 * Note: saved only local_kica_grade_items id & grade status from local_kica_grade_grades (calculated)
 * @param      $courseid
 * @param null $userid
 *
 * @throws \dml_exception
 */
function load_kica_grade_item_id_and_graded($courseid, $userid=null){
    if (!is_kica_exists()){
        return;
    }
    if (ned_global(['load_kica_grade_item_id_and_graded'])){
        return;
    }
    $sql =  '
        SELECT kgg.id, kgi.id as kgi_id, kgg.userid, kgg.knowledge, kgg.inquiry, kgg.communication, kgg.application
        FROM {local_kica_grade_items} AS kgi
        LEFT JOIN {local_kica_grade_grades} AS kgg ON (kgg.itemid = kgi.id AND kgg.courseid = kgi.courseid)
        WHERE kgi.courseid = :courseid AND kgg.id IS NOT NULL
        ';
    $params = ['courseid' => $courseid];
    if (!is_null($userid)){
        $sql .= ' AND kgg.userid = :userid';
        $params['userid'] = $userid;
    }

    $kica_info_records = _DB()->get_records_sql($sql, $params);
    if (empty($kica_info_records)){
        return;
    }

    $kica_graded = [];
    foreach ($kica_info_records as $r){
        if (!isset($kica_graded[$r->userid])){
            $kica_graded[$r->userid][0] = false;
        }
        $kica_graded[$r->userid][$r->kgi_id] = (!is_null($r->knowledge) || !is_null($r->inquiry) ||
            !is_null($r->communication) || !is_null($r->application));
    }

    // we can lose items, which has not grades yet, so...
    $sql =  '
        SELECT kgi.itemid, kgi.id
        FROM {local_kica_grade_items} AS kgi
        WHERE kgi.courseid = :courseid AND kgi.itemid <> 0
        ';
    $params = ['courseid' => $courseid];

    $kica_items = _DB()->get_records_sql($sql, $params);

    load_mod_grade_items($courseid, $userid);
    $gi_list = ned_global(['grade_items', $courseid]);
    $kica_itmes_id = [];

    foreach($gi_list as $gi_cat){
        foreach ($gi_cat as $gi){
            $kica_itmes_id[$gi->itemmodule][$gi->iteminstance] = isset($kica_items[$gi->id]) ? $kica_items[$gi->id]->id : 0;
        }
    }

    ned_global(['kica_item_id', $courseid], $kica_itmes_id);
    ned_global(['kica_is_graded'], $kica_graded);
    ned_global(['load_kica_grade_item_id_and_graded'], 1);
    if (is_null($userid)){
        ned_global(['load_kica_grade_item_id_and_graded_for_all_users'], 1);
    }
}

/**
 * Return saved groupid from block settings
 * @param $courseid
 *
 * @return false|int
 */
function get_cache_group($courseid){
    $context = \context_course::instance($courseid, MUST_EXIST);
    if (!has_capability('moodle/grade:viewall', $context)){
        return null;
    }
    $cache = \cache::make(PLUGIN_NAME, 'block_group');
    return $cache->get($courseid);
}

/**
 * Saved groupid in cache
 * @param $courseid
 * @param $groupid
 */
function set_cache_group($courseid, $groupid){
    if (is_null($groupid)){
        return;
    }
    $context = \context_course::instance($courseid, MUST_EXIST);
    if (!has_capability('moodle/grade:viewall', $context)){
        return;
    }
    $cache = \cache::make(PLUGIN_NAME, 'block_group');
    $cache->set($courseid, (int)$groupid);
}

/**
 * Check $groupid an replace it with the cache one if necessary
 * @param $groupid
 * @param $courseid
 * @param $groups
 *
 * @return false|int
 */
function check_group_with_cache($groupid, $courseid, $groups){
    if (is_null($groupid)){
        $cache_group = get_cache_group($courseid);
        $groupid = $cache_group;
    }
    if(!in_array($groupid, GROUPS, true) && !isset($groups[$groupid])){
        $groupid = null;
    }
    if ((is_null($groupid) || $groupid == GROUP_NONE) && count($groups) == 1){
        reset($groups);
        $groupid = key($groups);
    }
    return $groupid ?? GROUP_NONE;
}

/**
 * Check, can user see inactive people or not
 * @param $courseid
 *
 * @return bool
 * @throws \coding_exception
 */
function has_show_inactive_capability($courseid){
    $context = \context_course::instance($courseid, MUST_EXIST);
    return has_capability('moodle/course:viewsuspendedusers', $context);
}

/**
 * Return saved $show_inactive from block settings
 * @param $courseid
 *
 * @return false|int
 */
function get_cache_show_inactive($courseid){
    if (!has_show_inactive_capability($courseid)){
        return null;
    }
    $cache = \cache::make(PLUGIN_NAME, 'block_show_inactive');
    return $cache->get($courseid);
}

/**
 * Saved $show_inactive in cache
 * @param $courseid
 * @param $show_inactive
 */
function set_cache_show_inactive($courseid, $show_inactive){
    if (is_null($show_inactive) || !has_show_inactive_capability($courseid)){
        return;
    }
    $cache = \cache::make(PLUGIN_NAME, 'block_show_inactive');
    $cache->set($courseid, (int)$show_inactive);
}

/**
 * Check $show_inactive an replace it with the cache one if necessary
 *  Return 1/0 as $show_inactive value, null if user haven't capability to see inactive people
 * @param $show_inactive
 * @param $courseid
 *
 * @return null|int
 */
function check_show_inactive_with_cache($show_inactive, $courseid){
    if (!has_show_inactive_capability($courseid)){
        return null;
    }
    if (is_null($show_inactive)){
        $show_inactive = get_cache_show_inactive($courseid);
    }
    return (int)$show_inactive;
}
