<?php
/**
 * Lib of common functions for ned_teacher_tools, ned_student_menu, may be more later
 *
 * @package    block_ned_student_menu
 * @subpackage NED
 * @copyright  2020 NED {@link http://ned.ca}
 * @author     NED {@link http://ned.ca}
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

namespace block_ned_student_menu;
use block_ned_student_menu\shared_lib as SH;
use block_ned_teacher_tools\deadline_manager as DM;
use local_kica as KICA;
use local_ned_controller\assign_info;
use local_ned_controller\grade_info;
use local_ned_controller\marking_manager\marking_manager as MM;

const PLUGIN = 'ned_student_menu';

const PLUGIN_TYPE = 'block';
const PLUGIN_NAME = PLUGIN_TYPE . '_' . PLUGIN;
const PLUGIN_URL = '/blocks/' . PLUGIN . '/';
const PLUGIN_CAPABILITY = PLUGIN_TYPE . '/' . PLUGIN . ':';
const PLUGIN_PATH = __DIR__;
const DIRROOT = __DIR__ . '/../..';

const TT = 'ned_teacher_tools';
const TTX = 'ned_teacher_tools_x';
const SM = 'ned_student_menu';

const PLUGIN_TT = 'block_' . TT;
const PLUGIN_TTX = 'block_' . TTX;
const PLUGIN_SM = 'block_' . SM;

const is_TT = PLUGIN_NAME == PLUGIN_TT;
const is_TTX = PLUGIN_NAME == PLUGIN_TTX;
const is_aTT = is_TT || is_TTX;
const is_SM = PLUGIN_NAME == PLUGIN_SM;

const ACTIVITYSETTING = '1';
const PASSGRADEPERCENT = '2';

const RESUBMISSIONS_ALL = 0;
const RESUBMISSIONS_MADE_BY_RM = 10;
const RESUBMISSIONS_MADE_BY_RM_CONTROLLED_NOT_BY_RM = 11;
const RESUBMISSIONS_MADE_BY_RM_CONTROLLED_BY_RM = 12;
const RESUBMISSIONS_MADE_NOT_BY_RM = 20;
const RESUBMISSIONS_MADE_NOT_BY_RM_CONTROLLED_NOT_BY_RM = 21;
const RESUBMISSIONS_MADE_NOT_BY_RM_CONTROLLED_BY_RM = 22;
const RESUBMISSIONS_KEYS = [
    RESUBMISSIONS_ALL,
    RESUBMISSIONS_MADE_BY_RM, RESUBMISSIONS_MADE_BY_RM_CONTROLLED_NOT_BY_RM, RESUBMISSIONS_MADE_BY_RM_CONTROLLED_BY_RM,
    RESUBMISSIONS_MADE_NOT_BY_RM, RESUBMISSIONS_MADE_NOT_BY_RM_CONTROLLED_NOT_BY_RM, RESUBMISSIONS_MADE_NOT_BY_RM_CONTROLLED_BY_RM
];

const SEC_IN_DAY = 86400;

const SP_ACTION_REMOVESUBMISSION = 'removesubmissiondata';
const SP_ACTION_ALLOWRESUBMISSION = 'allowresubmission';
const SP_ACTIONS = [SP_ACTION_REMOVESUBMISSION, SP_ACTION_ALLOWRESUBMISSION];

const VIEW_USERS_SUSPENDED_NONE = 0;
const VIEW_USERS_SUSPENDED_ENROLLMENTS = 1;
const VIEW_USERS_SUSPENDED_ACCOUNTS = 2;
const VIEW_USERS_SUSPENDED_BOTH = 3;

const ROLE_OT = 'online-teacher';
const ROLE_CT = 'classroom-teacher';
const ROLE_STUDENT = 'student';

const GROUP_ALL = 0;
const GROUP_NONE = -1;
const GROUPS = [GROUP_ALL, GROUP_NONE];

const TAG_MIDTERM = 'Midterm point';
const TAG_FORMATIVE = 'Formative';
const TAG_SUMMATIVE =  'Summative';
const IMPORTANT_TAGS = [
    'midterm' => TAG_MIDTERM,
    'formative' => TAG_FORMATIVE,
    'summative' => TAG_SUMMATIVE,
];

require_once(DIRROOT . '/course/format/singleactivity/lib.php');
require_once(DIRROOT . '/lib/classes/user.php');
require_once(DIRROOT . '/lib/weblib.php');
require_once(PLUGIN_PATH . '/classes/activity_status.php');
require_once(PLUGIN_PATH . '/common_data_utils.php');

require_once(DIRROOT . '/mod/assign/locallib.php');

/**
 * @param \cm_info|\stdClass $activity
 * @param                    $userid
 * @param bool               $showkicagrades
 * @param bool               $deadlinemanagerinstalled
 * @param null|\stdClass     $rm_config
 * @param string             $mod_status = ''
 * @param array              $tags
 * @param \stdClass|null     $mm_data
 *
 * @return array of \html_table_row
 */
function render_activity_row($activity, $userid=null, $showkicagrades=false,
    $deadlinemanagerinstalled=false, $rm_config=null, $mod_status='', $tags=[], $mm_data=null) {
    global $USER, $DB, $COURSE, $OUTPUT, $PAGE;
    $userid = is_null($userid) ? $USER->id : $userid;
    $rows = [];
    $block_config = get_site_and_course_block_config($COURSE->id, PLUGIN);
    $course_context = \context_course::instance($COURSE->id);
    $add_row_main_class = [];
    $offset_row = 1;
    if (!($block_config->showduedatecolumn ?? true)){
        $offset_row = 0;
    }
    $now = time();
    // some short functions
    $format_submit_time = function($submit_time, $deadline_time=null, $draft=false){
        $now = (new \DateTime())->getTimestamp();
        if ($deadline_time &&
            (($submit_time && $submit_time > $deadline_time) || (!$submit_time && $now > $deadline_time))
        ){
            $title = $submit_time ? ned_date($submit_time) : str('duedate');
            $submit_time_f = fa('fa-calendar-times-o', '', $title);
        } elseif ($draft){
            $submit_time_f = status_img('saved');
        } elseif (!$submit_time){
            $submit_time_f = status_img('not_submitted');
        } else {
            $submit_time_f = ned_date($submit_time);
        }
        return $submit_time_f;
    };
    $format_grade_raw = function($status, $grade, $formative=false, $mm_data=null){
        if (!$formative && $grade != '-'){
            return $grade;
        }
        if ($mm_data){
            return status_img(get_status_by_mm_data($mm_data));
        }
        if ($status == 'waitingforgrade'){
            return status_img('waitingforgrade');
        }
        return status_img($status) ?: $grade;
    };
    $format_grade = function($status, $grade, $formative=false, $mm_data=null) use ($format_grade_raw){
        $f_grade = $format_grade_raw($status, $grade, $formative, $mm_data);
        if ($mm_data->overridden ?? false){
            $f_grade .= fa('fa-lock', '', str('gradeoverridden'));
        }

        return $f_grade;
    };

    $menu = new \action_menu();
    $menu->set_menu_trigger(fa('fa-caret-down'));
    $menu->attributes['class'] .= ' ned-actionmenu';
    $m_add = function($url, $url_params=[], $class='', $fa_icon=null, $text=null, $link_attr=[]) use (&$menu){
        $attr = $link_attr;
        $text = is_null($text) ? str($class) : $text;
        if ($fa_icon){
            $text = fa($fa_icon).$text;
        }
        $menu->add(link([$url, $url_params], $text, $class, $attr));
    };

    $main_tag = null;
    $important_tags = [];
    $params = [];
    $text_to_grade = '';
    $timer_span = '';
    $add_text_to_grade = function($last_attempt=true)
    use (&$OUTPUT, &$PAGE, &$COURSE, &$text_to_grade, &$menu, &$m_add, &$activity, &$important_tags, &$timer_span, &$course_context, &$userid){
        if (is_TT && $last_attempt){
            if ($activity->modname == 'assign'){
                $url = clone($PAGE->url);
                $url->param('cmid', $activity->id);

                foreach (SP_ACTIONS as $sp_action){
                    if ($sp_action == SP_ACTION_ALLOWRESUBMISSION && !in_array(TAG_SUMMATIVE, $important_tags)){
                        continue;
                    }
                    if (!SH::has_capability($sp_action, $course_context)){
                        continue;
                    }
                    $url->param('action', $sp_action);
                    $m_add($url, [], $sp_action);
                }
            }
            // just links
            $m_add("/mod/{$activity->modname}/view.php", ['id' => $activity->id, 'userid' => $userid, 'action' => 'grader', 'group' => 0],
                'moodlegradingpage', 'fa-table');
            $m_add(SH::MM_PAGE, ['courseid' => $COURSE->id, 'show' => 'marked', 'group' => 0, 'setuser' => $userid, 'mid' => $activity->id],
                'nedgradingpage', 'fa-table');
            $m_add("/mod/{$activity->modname}/view.php", ['id' => $activity->id], 'viewactivity', 'fa-external-link');
            $m_add("/course/modedit.php", ['update' => $activity->id], 'activitysettings', 'fa-cog');
        }

        if (!$menu->is_empty()){
            $text_to_grade .= $OUTPUT->render($menu);
        }

        if (!empty($timer_span) && $last_attempt){
            $text_to_grade .= $timer_span;
        }

        if (empty($text_to_grade)){
            return '';
        }
        $res = \html_writer::div($text_to_grade, 'text-to-grade');
        $text_to_grade = '';
        return $res;
    };

    if ($block_config->showformativesummativeicons ?? true) {
        if ($mm_data){
            foreach (IMPORTANT_TAGS as $tag_key => $tag){
                $is_tag = 'is_'.$tag_key;
                if ($mm_data->$is_tag ?? false){
                    $main_tag = $main_tag ?? $tag;
                    $important_tags[] = $tag;
                }
            }
        } else {
            foreach (IMPORTANT_TAGS as $tag){
                if (in_array($tag, $tags) !== false){
                    $main_tag = $main_tag ?? $tag;
                    $important_tags[] = $tag;
                }
            }
        }
    }

    if ($main_tag){
        $params = ['data-tags' => $main_tag, 'title' => $main_tag];
    }

    // deadline
    $finaldeadline = null;
    $dm_module = $deadlinemanagerinstalled ? DM::get_dmm_by_cm($activity) : null;
    if ($mm_data){
        $finaldeadline = $mm_data->cutoffdate ?? null;
    } elseif ($dm_module) {
        $finaldeadline = $dm_module->get_user_final_deadline($userid);
    } else {
        $inst = $DB->get_record($activity->modname, ['id' => $activity->instance], '*');
        $finaldeadline = $inst->duedate ?? null;
    }
    $text_2 = ned_date($finaldeadline);
    if ($finaldeadline && $finaldeadline > $now){
        $timer_span = \html_writer::span('', 'timer duedate-time',
            ['data-datetime' => $finaldeadline, 'title' => str('duedate')]);
    }

    // extension
    if (is_TT && !empty($dm_module)) {
        $extension_course_data = DM::get_extension_course_data($activity->course, $activity->context);
        list($numberofextensions, $can_add_extension, $showextensionicon) = $dm_module->get_extension_data($userid, ...$extension_course_data);

        $text_2 .= $OUTPUT->render_from_template('block_ned_teacher_tools/deadline_manager_user_extension_button', [
            'numberofextensions' => $numberofextensions,
            'showextensionicon' => $showextensionicon,
            'can_add_extension' => $can_add_extension,
            'userid' => $userid,
            'cmid' => $activity->id,
        ]);
    }

    // activity name & tags
    $activity_name_add = '';
    if ($main_tag == TAG_MIDTERM){
        $add_row_main_class[] = 'row-midterm-point';
    }

    $output = row(array_fill(0, $offset_row + 4, ''));
    $output->cells[0] = cell($activity_name_add . mod_link($activity),
        'activity-name add-tag-icon', $params);

    if (!empty($offset_row)){
        $output->cells[1] = cell($text_2, 'activity-type');
    }

    // submissions
    $submission_info = new assign_info($activity, $userid);
    if ($submission_info->exist && $submission_info->submissions_count > 0){
        if (empty($offset_row)){
            $output->cells[0]->rowspan =  $submission_info->submissions_count;
        } else {
            $output->cells[0]->rowspan = $output->cells[1]->rowspan = $submission_info->submissions_count;
        }
        // it's need to sorting, otherwise we can make just empty cells here
        if (empty($offset_row)){
            $hidden_cells = [cell($output->cells[0]->text, 'display-none')];}
        else {
            $hidden_cells = [cell($output->cells[0]->text, 'display-none'), cell($output->cells[1]->text, 'display-none')];
        }
        $submited_sort = 0;
        $graded_sort = 0;
        $grade_sort = 0;
        $rm_is_on = $rm_config && isset($rm_config->assignments[$activity->id]) && $rm_config->assignments[$activity->id];
        foreach ($submission_info->submissions as $num => $submission){
            $grade_params = [];
            $add_row_class = $add_row_main_class;
            $add_row_attr = [];
            $is_last_row = $num == ($submission_info->submissions_count - 1);
            $is_first_row = $num == 0;

            if ($is_first_row){
                $row = $output;
                if (empty($offset_row)) {
                    $row->cells = [$output->cells[0]];
                } else {
                    $row->cells = [$output->cells[0], $output->cells[1]];
                }
            } else {
                $row = row($hidden_cells);
            }

            if ($is_last_row){
                $submit_time = $submission_info->last_submit_time;
                $grade_time = $submission_info->last_grade_time;
                $rawgrade = $submission_info->last_grade;
            } else {
                $submit_time = $submission->submitted ? $submission->submit_time : null;
                $grade_time = $submission->graded ? $submission->grade_time : null;
                $rawgrade = $submission->graded ? $submission->grade : null;
            }

            list($g_percenta, $g_format, $contribution, $status, $gradetimecreated, $fgrade_time) =
                get_formatted_grade_by_grade_item($userid, $submission_info->grade_items, $showkicagrades, $rawgrade,
                    null, null, null, null, $is_last_row);

            if ($is_last_row){
                $submited_sort = (int)$submit_time;
                $graded_sort = (int)$grade_time;
                $grade_sort = (int) $g_percenta;
                $submit_time_f = $format_submit_time($submit_time, $finaldeadline, $submission->draft);
                $status = $mm_data ? get_status_by_mm_data($mm_data) : $mod_status;
                if ($submit_time){
                    $add_row_class[] = 'submitted';
                }
                if ($mm_data->unmarked ?? false){
                    $add_row_class[] = 'unmarked';
                }
            } else {
                $add_row_class[] = 'previous-attempt';
                $submit_time_f = ned_date($submit_time);
            }

            if (!$is_first_row){
                $add_row_class[] = 'resubmission-attempt';
                if (!$submit_time && $finaldeadline && $now > $finaldeadline){
                    $add_row_class[] = 'resubmission-due';
                }
            }

            if ($submit_time && $g_format == '-'){
                if ($mm_data){
                    $grade_time_f = '-';
                    $g_format = $format_grade($status, status_img($status), false, $mm_data);
                } else {
                    $grade_time_f = status_img('submitted');
                    $g_format = $format_grade($status, $grade_time_f, false, $mm_data);
                }
                $graded_sort = $grade_sort = 0;
            } else {
                $g_format = $format_grade($status, $g_format, $main_tag == TAG_FORMATIVE, $is_last_row ? $mm_data : null);
                $grade_time_f = ned_date($grade_time);
            }

            if ($rm_is_on){
                $ri_class = 'resubmission-info rm-i ';
                $add_button = false;
                $add_time = false;
                $add_remove_link = false;
                $show_rm = true;
                $title = '';
                if (!$is_last_row){
                    if ($submission_info->submissions[$num+1]->user_sign){
                        $ri_class .= 'rm-done';
                        $title = str('resubmission_legend:done');
                    } else {
                        $ri_class .= 'rm-free';
                        $title = str('resubmission_legend:free');
                    }
                } else {
                    if (!$submit_time && !$submission->draft && !$submission_info->last_grade_time){
                        if (!$is_first_row){
                            $ri_class = 'resubmission-info rm-remove';
                            $add_remove_link = true;
                        }
                    }
                    if (!$submission_info->last_grade_time){
                        if ($submit_time){
                            $ri_class = '';
                        } else {
                            if (!$add_remove_link){
                                $ri_class = 'rm-clock';
                                $title = str('resubmission_legend:clock');
                            }
                        }
                    } elseif ($rm_config->max_time && ($rm_config->now > ($submission_info->last_grade_time + $rm_config->max_time))){
                        $ri_class .= 'rm-lock';
                        $title = str('resubmission_legend:lock');
                    } else {
                        $can_ressubmit = $rm_config->can_ressubmit &&
                            !($rm_config->max_attempts && $rm_config->max_attempts <= $submission_info->user_signed_submissions_count);
                        if ($can_ressubmit){
                            $add_time = true;
                            $errors = $rm_config->pp_error;
                            if (!empty($errors)){
                                $ri_class .= 'rm-not-allowed';
                                $title = join('&#13;', $errors);
                            } else {
                                $ri_class .= 'rm-unlock';
                                $title = str('resubmit_title');
                                $add_button = true;
                            }
                        } else {
                            $show_rm = false;
                        }
                    }
                }

                if ($show_rm){
                    $add_text = '';
                    if (is_SM && $add_button){
                        $input_params = ['type' => 'submit', 'id' => 'id_submitbutton', 'value' => str('resubmit'),
                            'href' => new \moodle_url('/blocks/ned_student_menu/ask_to_resubmit.php', ['id' => $activity->id])];
                        $add_text .= \html_writer::empty_tag('input', $input_params);
                    }
                    if ($add_time){
                        $timer_span = \html_writer::span('', 'timer resubmission-time',
                            ['data-grade_time' => $submission_info->last_grade_time, 'title' => str('resubmit_time_title')]);
                    }
                    if (is_SM && $add_remove_link){
                        $add_text .= link([PLUGIN_URL . 'unused_resubmission_remove.php', ['cmid' => $activity->id, 'userid' => $userid]],
                            ' ', 'btn btn-primary fa fa-trash', ['title' => str('cancelresubmissionrequest')]);
                    }
                    $text_to_grade .= \html_writer::div($add_text, $ri_class, ['title' => $title]);
                }
            }

            $row->cells[$offset_row + 1] = cell($submit_time_f, 'activity-submitted');
            $row->cells[$offset_row + 2] = cell($grade_time_f,'activity-graded');
            $row->cells[$offset_row + 3] = cell($g_format.$add_text_to_grade($is_last_row), 'activity-grade '.$status, $grade_params);

            $row->attributes['class'] .= join(' ', $add_row_class);
            $row->attributes = array_merge($row->attributes, $add_row_attr);
            $rows[$num] = $row;
        }
        // add for all cells of one activity the same sort order (of last submission)
        foreach ($rows as $row){
            $row->cells[$offset_row + 1]->attributes['data-sort-value'] = $submited_sort;
            $row->cells[$offset_row + 2]->attributes['data-sort-value'] = $graded_sort;
            $row->cells[$offset_row + 3]->attributes['data-sort-value'] = $grade_sort;
        }

    } else {
        list($g_percenta, $g_format, $contribution, $status, $g_create, $g_time) =
            get_formatted_grade($activity, $userid, null, $showkicagrades, $mm_data);

        if ($mm_data || !is_null($g_create)){
            $submit_time_raw = $mm_data->submit_time ?? ($g_create ?: 0);
            $submit_time_f = $format_submit_time($submit_time_raw, $finaldeadline);
            $status = $mm_data ? $mod_status : $status;
        } else {
            $submit_time_raw = 0;
            $submit_time_f = '-';
        }

        $g_class = ['activity-grade', $status];
        if ($submit_time_raw){
            $add_row_main_class[] = 'submitted';
        }
        if ($mm_data){
            if ($mm_data->unmarked ?? false){
                $add_row_main_class[] = 'unmarked';
            }
            if ($mm_data->shouldpass ?? false){
                switch ($status){
                    case 'graded_kica_summative':
                    case 'graded_kica_formative':
                    case 'completed':
                        $g_class[] = 'passing';
                        break;
                    case 'incompleted_summative':
                    case 'incompleted_formative':
                    case 'incompleted':
                    case 'kica_zerograde':
                        $g_class[] = 'failing';
                        break;
                }
            }
        }
        $g_format = $format_grade($mod_status, $g_format, $main_tag == TAG_FORMATIVE, $mm_data);
        $grade_time_f = ned_date($g_time);
        $output->cells[$offset_row + 1] = cell($submit_time_f, 'activity-submitted', ['data-sort-value' => (int)$submit_time_raw]);
        $output->cells[$offset_row + 2] = cell($grade_time_f, 'activity-graded', ['data-sort-value' => (int)$g_time]);
        $output->cells[$offset_row + 3] = cell($g_format.$add_text_to_grade(), $g_class, ['data-sort-value' => $g_percenta]);

        $output->attributes['class'] .= join(' ', $add_row_main_class);
        $rows[0] = $output;
    }


    return $rows;
}

/**
 * @param                  $userid
 * @param                  $gradeitem
 * @param bool             $showkicagrades
 * @param null|\int|\float $finalgrade
 * @param \mixed           $reporttable
 * @param null             $gradetimecreated
 * @param null             $gradetime
 * @param \stdClass|null   $mm_data
 * @param bool             $final_attempt
 *
 * @return array [$gradepercentage, $gradeformatted, $contributiontocoursetotal, $status, $gradetimecreated, $gradetime]
 */
function get_formatted_grade_by_grade_item($userid, $gradeitem, $showkicagrades=false, $finalgrade=null, $reporttable=null,
    $gradetimecreated=null, $gradetime=null, $mm_data=null, $final_attempt=true){
    global $DB, $USER;

    $gradepercentage = 0;
    $gradeformatted = '-';
    $contributiontocoursetotal = '-';
    $status = 'notgraded';
    $default = [$gradepercentage, $gradeformatted, $contributiontocoursetotal, $status, $gradetimecreated, $gradetime];

    if (!$gradeitem) return $default;
    $grademax = $gradeitem->grademax;

    if (!$userid) {
        $userid = $USER->id;
    }
    if (!is_kica_exists()){
        $showkicagrades = false;
    }
    $config = get_config(is_aTT ? PLUGIN_NAME : PLUGIN_TT);
    $get_usual_grades = !$showkicagrades;
    $kica = $grade = $kicaitem = null;

    if ($showkicagrades) {
        $kica = $DB->get_record('local_kica', array('courseid' => $gradeitem->courseid));
        $kicaitem = new KICA\kica_item(['id' => $gradeitem->id], true, true);
        if ($kica && $kicaitem->id){
            $kicagrade = new KICA\grade($userid, $kicaitem->id);
            $grade = $kicagrade->get_grade();
            $default[4] = $gradetimecreated = $kicagrade->timecreated;
            $default[5] = $gradetime = $kicagrade->timemodified;
            $finalgrade = $grade->finalgrade;
            if (is_null($finalgrade)) return $default;
            $grademax = $kicaitem->knowledge + $kicaitem->inquiry + $kicaitem->communication + $kicaitem->application;
        } else {
            $get_usual_grades = true;
        }
    }

    if ($get_usual_grades) {
        if ($showkicagrades){
            if ($mm_data){
                $default[4] = $gradetimecreated = $mm_data->submit_time;
                $default[5] = $gradetime = $mm_data->grade_time;
                $finalgrade = $mm_data->finalgrade;
            } else {
                $grade = grade_info::get_grade_grades_s($gradeitem->id, $userid);
                if ($grade){
                    $default[4] = $gradetimecreated = $grade->timecreated;
                    $default[5] = $gradetime = $grade->timemodified;
                    $finalgrade = $grade->finalgrade;
                }
            }
        }

        if ($gradeitem->gradetype == GRADE_TYPE_SCALE) {
            $gradeformatted =  '-';
        } else if ($gradeitem->gradetype == GRADE_TYPE_VALUE) {
            $default[1] = $gradeformatted =  $grademax > 0 ? '/' . round($grademax) : '-';
        }

        if (is_null($finalgrade)) return $default;

        if ($gradeitem->gradetype == GRADE_TYPE_SCALE) {
            if (!empty($gradeitem->scaleid)) {
                $scale = grade_info::get_scale_s($gradeitem->scaleid);
                $gradeval = (int)$finalgrade; // scales use only integers
                $scales = explode(",", $scale->scale);
                $gradeformatted = $scales[($gradeval-1)];
                $gradepercentage = -$gradeval;
            }
        }

        if ($reporttable) {
            foreach ($reporttable as $item) {
                if (isset($item['contributiontocoursetotal'])
                    && (strpos($item['contributiontocoursetotal']['headers'], 'row_'.$gradeitem->id.'_'.$userid) !== false))
                    $contributiontocoursetotal = $item['contributiontocoursetotal']['content'];
            }
        }
    }

    // TODO Find the way to get KICA grades for previous submissions
    if ($showkicagrades && !$final_attempt) {
        $default[1] = '';
        return $default;
    }

    $show_as_kica = $showkicagrades && !$get_usual_grades;

    if ($show_as_kica || $gradeitem->gradetype == GRADE_TYPE_VALUE){
        if ($grademax > 0){
            $finalgradeold = '';
            if ($grade->reduction ?? false) {
                $finalgradeold = \html_writer::tag('del', $grade->finalgrade_orig, ['class' => 'dimmed_text']) . ' ';
            }
            $gradepercentage = round(($finalgrade / $grademax) * 100);
            $gradeformatted = $finalgradeold . round($finalgrade) . '/' . round($grademax) . ' (' . $gradepercentage . '%)';
        } else {
            $gradepercentage = 1;
            $gradeformatted = '-';
        }

        if ($config->passgrade == ACTIVITYSETTING) {
            if ($gradeitem->gradepass > 0) {
                if ($gradeitem->gradepass <= $finalgrade) {
                    $status = 'passing';
                } else {
                    $status = 'failing';
                }
            };
        } else if ($config->passgrade == PASSGRADEPERCENT) {
            if ($config->passgradepercent <= $gradepercentage) {
                $status = 'passing';
            } else {
                $status = 'failing';
            }
        }
    }

    if ($contributiontocoursetotal == '0.00 %') {
        $contributiontocoursetotal = '-';
    }
    return [$gradepercentage, $gradeformatted, $contributiontocoursetotal, $status, $gradetimecreated, $gradetime];
}

/**
 * @param \cm_info|\stdClass $mod
 * @param int                $userid
 * @param null               $reporttable
 * @param bool               $showkicagrades
 * @param \stdClass|null     $mm_data
 *
 * @return array [$gradepercentage, $gradeformatted, $contributiontocoursetotal, $status, $gradetimecreated, $gradetime]
 */
function get_formatted_grade($mod, $userid=0, $reporttable=null, $showkicagrades=false, $mm_data=null) {
    global $USER;

    if (!$userid) {
        $userid = $USER->id;
    }
    $gradetimecreated = $gradetime = $finalgrade = null;
    $gradeitem = grade_info::get_grade_items_by_mod_s($mod);
    if ($gradeitem){
        if (!$showkicagrades){
            if ($mm_data){
                $gradetimecreated = $mm_data->submit_time;
                $gradetime = $mm_data->grade_time;
                $finalgrade = $mm_data->finalgrade;
            } else {
                $grade = grade_info::get_grade_grades_s($gradeitem->id, $userid);
                if ($grade){
                    $gradetimecreated = $grade->timecreated;
                    $gradetime = $grade->timemodified;
                    $finalgrade = $grade->finalgrade;
                }
            }
        }
    }

    list($gradepercentage, $gradeformatted, $contributiontocoursetotal, $status, $gradetimecreated, $gradetime) =
        get_formatted_grade_by_grade_item($userid, $gradeitem, $showkicagrades, $finalgrade, $reporttable, $gradetimecreated, $gradetime, $mm_data);
    return array($gradepercentage, $gradeformatted, $contributiontocoursetotal, $status, $gradetimecreated, $gradetime);
}

/**
 * @param $courseid
 * @param $userid
 *
 * @return \cm_info[]
 */
function get_gradable_activities($courseid, $userid) {
    static $_data = [];

    if (!isset($_data[$courseid][$userid])){
        $modinfo = get_fast_modinfo($courseid, $userid);
        $activities = $modinfo->get_cms();

        foreach ($activities as $key => $mod) {
            $modulecontext = \context_module::instance($mod->id);
            if (!$mod->visible && !has_capability('moodle/course:viewhiddenactivities', $modulecontext, $userid) ||
                !has_capability('mod/assign:submit', $modulecontext, $userid)) {
                unset($activities[$key]);
                continue;
            }
            $instance = get_module_db($mod->course, $mod->modname, $mod->instance);
            if (!$instance || (($mod->modname == 'forum') && !($instance->assessed > 0))) {
                unset($activities[$key]);
                continue;
            }
            if (!$item = get_grade_item($courseid, $mod->modname, $mod->instance)) {
                unset($activities[$key]);
                continue;
            }
        }
        $_data[$courseid][$userid] = $activities;
    }

    return $_data[$courseid][$userid];
}

/**
 * Use instead of get_gradable_activities to get activities for Student/Class progress
 *
 * @param       $courseid
 * @param       $userid
 * @param array $compare_list
 *
 * @return \cm_info[]
 */
function get_important_activities_by_compare_list($courseid, $userid, $compare_list=[]) {
    static $_data = [];

    if (empty($compare_list)){
        return [];
    }

    if (!isset($_data[$courseid][$userid])){
        $modinfo = get_fast_modinfo($courseid, $userid);
        $activities = $modinfo->get_cms();
        $result = [];

        foreach ($activities as $key => $mod) {
            if (!$mod->is_visible_on_course_page()) continue;
            if (!isset($compare_list[$key])) continue;

            $modulecontext = \context_module::instance($mod->id);
            if (!$mod->visible && !has_capability('moodle/course:viewhiddenactivities', $modulecontext, $userid) ||
                !has_capability('mod/assign:submit', $modulecontext, $userid)) {
                continue;
            }

            $result[$key] = $mod;
        }
        $_data[$courseid][$userid] = $result;
    }

    return $_data[$courseid][$userid];
}

/**
 * @param $assignid
 *
 * @return bool
 * @throws \dml_exception
 */
function is_offline_assign($assignid) {
    global $DB;

    $sql = /** @lang text */
        "SELECT 1
              FROM {assign_plugin_config} pc
             WHERE pc.assignment = ?
               AND pc.subtype = 'assignsubmission' 
               AND pc.value = 1 
               AND pc.plugin <> 'comments'";

    return !$DB->record_exists_sql($sql, array($assignid));
}

/**
 * @param $mod
 * @param $userid
 *
 * @return bool
 * @throws \dml_exception
 */
function assignment_status($mod, $userid) {
    global $CFG, $DB, $SESSION;

    require_once($CFG->dirroot . '/mod/assign/locallib.php');

    if (isset($SESSION->completioncache)) {
        unset($SESSION->completioncache);
    }

    if ($mod->modname == 'assignment') {
        return false;
    } else if ($mod->modname == 'assign') {
        if (!($assignment = get_module_db($mod->course, $mod->modname, $mod->instance))) {
            return false;
        }

        $offlineassignment = is_offline_assign($assignment->id);

        if (!$submission = $DB->get_records('assign_submission', array('assignment' => $assignment->id,
            'userid' => $userid), 'attemptnumber DESC', '*', 0, 1)) {
            $submission = new \stdClass();
            $submission->status = ASSIGN_SUBMISSION_STATUS_NEW;
            $submission->timemodified = 0;
            $submission->attemptnumber = 0;
            $attemptnumber = 0;
        } else {
            $submission = reset($submission);
            $attemptnumber = $submission->attemptnumber;
            if (($submission->status == 'reopened') && ($submission->attemptnumber > 0)) {
                $attemptnumber = $submission->attemptnumber - 1;
            }
        }

        if ($submissionisgraded = $DB->get_records('assign_grades',
            array('assignment' => $assignment->id, 'userid' => $userid, 'attemptnumber' => $attemptnumber),
            'attemptnumber DESC', '*', 0, 1)) {
            $submissionisgraded = reset($submissionisgraded);
            if ($submissionisgraded->grade > -1) {
                if (($submission->timemodified > $submissionisgraded->timemodified)
                    || ($submission->attemptnumber > $submissionisgraded->attemptnumber)) {
                    $graded = false;
                } else {
                    $graded = true;
                }
            } else {
                $graded = false;
            }
        } else {
            list($gradepercentage, $gradeformatted, $contributiontocoursetotal, $status) = get_formatted_grade($mod, 0);
            if ($status !== 'notgraded') {
                $graded = true;
            } else {
                $graded = false;
            }
        }

        if ($submission->status == ASSIGN_SUBMISSION_STATUS_DRAFT) {
            if ($graded) {
                return 'submitted';
            } else {
                return 'saved';
            }
        } else if ($submission->status == ASSIGN_SUBMISSION_STATUS_REOPENED) {
            return 'submitted';
        } else if ($submission->status == ASSIGN_SUBMISSION_STATUS_SUBMITTED) {
            if ($graded) {
                return 'submitted';
            } else {
                return 'waitinggrade';
            }
        } else if ($submission->status == ASSIGN_SUBMISSION_STATUS_NEW) {
            if ($graded) {
                if ($offlineassignment) {
                    return 'offlinegraded';
                } else {
                    return 'gradeoverride';
                }
            }
        }
    }

    return false;
}

/**
 * @param $itemid
 * @param $userid
 *
 * @return bool|float
 * @throws \dml_exception
 */
function gradebook_grade ($itemid, $userid) {
    if ($grade = get_grade_grade ($userid, $itemid)) {
        return $grade->finalgrade;
    } else {
        return false;
    }
}

/**
 * Return activity_status with counted activities if $return_object, otherwise status list of activities
 * @param      $activities
 * @param int  $userid
 * @param bool $return_object
 *
 * @return array|activity_status
 */
function get_activity_status_numbers($activities, $userid=0, $return_object=false) {
    global $USER;
    if (!$userid) {
        $userid = $USER->id;
    }
    $user_activity_status = new activity_status();
    if (!empty($activities)){
        $mod = reset($activities);
        $options = ['courseid' => $mod->course, 'userid' => $userid];
        $activity_completion = get_course_modules_completion($options, true, true);
        foreach ($activities as $mod) {
            $user_activity_status->check_activity($mod, $userid,
                false, null, null, $activity_completion[$mod->id] ?? false);
        }
    }

    return $return_object ? $user_activity_status : $user_activity_status->get_list();
}

/**
 * @param $courseid
 * @param $userid
 *
 * @return float|string
 */
// TODO load (count) all of them, if we need all user grades form course
function get_course_grade($courseid, $userid){
    if (is_kica_exists() && utils::kica_gradebook_enabled($courseid)){
        $finalgrade = KICA\helper::get_course_average($userid, $courseid);
    } else {
        $courseitem = \grade_item::fetch_course_item($courseid);
        $coursegrade = new \grade_grade(array('itemid' => $courseitem->id, 'userid' => $userid));
        $coursegrade->grade_item =& $courseitem;
        $finalgrade = $coursegrade->finalgrade;
    }
    return is_null($finalgrade) ? '-' : round($finalgrade, 2);
}

/**
 * Get options for RM from block and site config
 * @param \stdClass $tt_config
 *
 * @return \stdClass|null
 * @throws \Exception
 */
function get_rm_config(\stdClass $tt_config=null){
    if (!$tt_config->enableresubmissions || empty($tt_config->resubmission_assignments)){
        return null;
    }

    $rm_config = new \stdClass();
    $rm_config->max_attempts = $tt_config->maximumattempts;
    $rm_config->max_course_attempts = $tt_config->maximumattemptsincourse;
    $rm_config->max_time = $tt_config->daysforresubmission * 24 * 3600;
    $rm_config->assignments = $tt_config->resubmission_assignments;
    $rm_config->now = (new \DateTime())->getTimestamp();
    $rm_config->pp_error = [];

    $rm_config->show_pp = false;
    $rm_config->can_ressubmit = false;
    // resubmissioncriteria: 0 - if none, 1 - if need check participation_power
    $rm_config->check_pp = (bool) $tt_config->resubmissioncriteria;
    $rm_config->need_pp = $tt_config->participationpowerrequired;

    return $rm_config;
}

/**
 * Get options for RM from block and site config by course id
 * @param $courseid
 *
 * @return \stdClass|null
 */
function get_rm_config_by_courseid($courseid){
    $tt_config = get_site_and_course_block_config($courseid, TT);
    $tt_config->resubmission_assignments =
        ($tt_config->enableresubmissions && isset($tt_config->resubmission_assignments)) ? $tt_config->resubmission_assignments : [];
    return get_rm_config($tt_config);
}

/**
 * Update RM options for user by its submissions count & participation power
 *
 * @param \stdClass $rm_config
 * @param int       $submissions_count
 * @param null      $participation_power
 *
 * @return \stdClass|null
 */
function update_rm_config(\stdClass $rm_config, $submissions_count=0, $participation_power=null){
    if (!$rm_config){
        return null;
    }

    $rm_config->can_ressubmit = !($rm_config->max_course_attempts && $rm_config->max_course_attempts <= $submissions_count);
    if ($rm_config->check_pp && !is_null($participation_power)){
        $rm_config->show_pp = true;
        if ($participation_power < $rm_config->need_pp){
            $rm_config->pp_error[] = str('resubmit_pp_error', [$participation_power, $rm_config->need_pp]);
        }
    }

    return $rm_config;
}

/**
 * Return config for block on course page by course id
 * Don't use it in a course loop!
 *
 * @param        $course_id
 *
 * @param string $plugin_name
 *
 * @return bool|\stdClass
 * @throws \dml_exception
 */
function get_block_config($course_id, $plugin_name='ned_student_menu'){
    global $DB;
    $G_NAME = 'block_configs';
    $block_config = ned_global([$G_NAME, $course_id, $plugin_name]);

    if (is_null($block_config)){
        $context = \context_course::instance($course_id);
        $blockdata = $DB->get_record('block_instances', ['blockname' => $plugin_name, 'parentcontextid' => $context->id]);
        if (!$blockdata){
            $block_config = false;
        } else {
            $config = unserialize(base64_decode($blockdata->configdata));
            if (!$config){
                $config = new \stdClass();
            }
            $config->block_id = $blockdata->id;
            $config->course_id = $course_id;
            $block_config = $config;
        }
        ned_global([$G_NAME, $course_id, $plugin_name], $block_config);
    }

    return $block_config;
}

/**
 * Save config of block for course page by course id
 * Don't use it in a course loop!
 *
 * @param        $course_id
 * @param string $plugin_name
 * @param        $config
 *
 * @return bool
 */
function save_block_config($course_id, $plugin_name='ned_student_menu', $config=null){
    global $DB;
    $G_NAME = 'block_configs';
    if (!$config){
        $block_config = ned_global([$G_NAME, $course_id, $plugin_name]);
        if (!$block_config){
            return false;
        }
    } else {
        $block_config = $config;
    }

    $configdata = base64_encode(serialize($block_config));
    $context = \context_course::instance($course_id);
    if ($DB->set_field('block_instances', 'configdata', $configdata,
        ['blockname' => $plugin_name, 'parentcontextid' => $context->id])){
        ned_global([$G_NAME, $course_id, $plugin_name], $block_config);
        return true;
    }

    return false;
}

/**
 * Get block config for course, based on site block config and course block settings
 *
 * @param        $course_id
 * @param string $plugin_name
 *
 * @return object
 */
function get_site_and_course_block_config($course_id, $plugin_name='ned_teacher_tools_x'){
    static $_data = [];
    if (!isset($_data[$course_id][$plugin_name])){
        $site_config = get_config('block_' . $plugin_name);
        $block_config = get_block_config($course_id, $plugin_name);
        $block_config = $block_config ? $block_config : [];
        $config = (object)array_merge((array)$site_config, (array)$block_config);
        $_data[$course_id][$plugin_name] = $config;
    }

    return $_data[$course_id][$plugin_name];
}


/**
 * Return div block with activity links to Student progress page
 * @param $activity_stats
 * @param $base_url
 * @param $url_params
 *
 * @return string
 */
function get_activity_menu($activity_stats, $base_url, $url_params){
    list($completed, $incompleted, $saved, $notattempted, $waitingforgrade) = $activity_stats;

    $link = function($show, $number_activity, $key_activity=null) use ($base_url, $url_params){
        $key_activity = $key_activity ? $key_activity : $show;
        $m_url = new \moodle_url($base_url, array_merge($url_params, ['show' => $show]));
        $text = $number_activity .' '. str($key_activity);
        return \html_writer::link($m_url, $text);
    };

    $img = function($filename, $class='icon') {
        return img($filename, $class, PLUGIN_NAME);
    };

    $div = function($image='', $num=0, $show='', $key_activity=null, $add_class='') use ($img, $link){
        $show = empty($show) ? $image : $show;
        $image = empty($image) ? '' : $img($image);
        $text = $link($show, $num, $key_activity);

        return \html_writer::div($image . $text, 'ned-activity-link ' . $add_class);
    };

    $content = '';

    // Completed.
    if (!is_null($completed)) {
        // Incomplete.
        if ($incompleted) {
            $content .= $div('completed', $completed, 'completed', 'completedsuccessful');
            $content .= $div('incomplete', $incompleted, 'incompleted', 'completedunsuccessful');
        } else {
            $content .= $div('completed', $completed);
        }

    }

    // Draft.
    if (!is_null($saved) && $saved) {
        $content .= $div('saved', $saved, 'draft');
    }

    // Not Attempted.
    if (!is_null($notattempted)) {
        $content .= $div('notattempted', $notattempted);
    }
    // Waiting for grade.
    if (!is_null($waitingforgrade)) {
        $content .=  $div('unmarked', $waitingforgrade, 'waitingforgrade');
    }

    return $content;
}

/**
 * Create tag ulr-select
 *
 * @param $url
 * @param $tags
 * @param $tag
 *
 * @return string
 */
function tags_print_course_menu($url, $tags, $tag){
    global $OUTPUT;

    $def_tags = ['0' => str('alltags'),
        '-1' => str('nonetags')];
    ksort($tags);
    $select = new \single_select($url, 'tag', $def_tags + $tags, $tag, null, 'selecttag');
    $select->label = \html_writer::tag('i','',
        ['class' => 'fa fa-tags', 'title' => str('choosetag')]);
    $output = \html_writer::div($OUTPUT->render($select), 'groupselector tag-selector');

    return $output;
}

/**
 * Render student progress page, return nothing
 *  none capability checks, do it before, if you need
 * @param        $course
 * @param        $user
 * @param        $show
 * @param        $fs_type
 * @param null   $groupid
 * @param null   $tag
 * @param string $studentselector
 * @param string $extra
 *
 * @throws \coding_exception
 * @throws \dml_exception
 * @throws \moodle_exception
 */
function render_student_progress_page($course, $user, $show, $fs_type, $groupid=null, $tag=null, $studentselector='', $extra=''){
    global $OUTPUT, $PAGE, $CFG;

    if (!$course){
        \print_error('There is wrong course!');
    }

    $courseid = $course->id;
    $userid = $user ? $user->id : null;
    $show_keys = [
        activity_status::STATUS_ALL, activity_status::STATUS_COMPLETED, activity_status::STATUS_INCOMPLETED,
        activity_status::STATUS_DRAFT, activity_status::STATUS_NOTATTEMPTED, activity_status::STATUS_WAITINGFORGRADE
    ];
    $show = isset_in_list($show_keys, $show, activity_status::STATUS_ALL);
    $fs_type = isset_in_list([SUMMATIVE_ACTIVITY, FORMATIVE_ACTIVITY], $fs_type, USUAL_ACTIVITY);
    $add_body_class = 'ned-student-progress path-blocks-NED';

    $base_url = PLUGIN_URL . 'student_progress.php';
    $base_url_param = ['courseid' => $courseid, 'show' => $show, 'fs_type' => $fs_type];

    if(is_aTT){
        $base_url_param = array_merge($base_url_param, ['group' => $groupid, 'user' => $userid, 'tag' => $tag, 'prevpage'=>'student_progress']);
    }

    $called_from_somewhere = $PAGE->has_set_url();
    $result = '';

    $print_result = function ($result) use ($OUTPUT, $course, $called_from_somewhere, $extra) {
        if (!$called_from_somewhere){
            echo $OUTPUT->header();
        }

        echo $extra;
        echo \html_writer::div($result,'', ['id' => 'mark-interface']);

        if (!$called_from_somewhere){
            echo $OUTPUT->footer($course);
        }
    };

    // Page settings
    if (!$called_from_somewhere){
        $PAGE->set_url($base_url, $base_url_param);
        $PAGE->navbar->add(str('studentprogress'),
            new \moodle_url($base_url, array_diff_key($base_url_param, ['show' => 0, 'fs_type' => 0, 'tag' => 0])));
        if (is_aTT && $userid){
            $PAGE->navbar->add(fullname($user), new \moodle_url('/user/view.php', ['id' => $userid, 'course' => $courseid]));
        }
        $title = str('studentprogress');
        $PAGE->set_title($title . '');
        $PAGE->set_heading($course->fullname . ': ' . $title);
        $PAGE->add_body_class($add_body_class);
    } else {
        $result .= \html_writer::script("window.document.body.className += ' $add_body_class';");
    }

    if (!$userid || $groupid == GROUP_NONE){
        $result .= \html_writer::div(str('noactivity'), 'no-activity');
        $print_result($result);
        return;
    }

    // should be checked ASAP after PAGE is ready, as we may redirected from here
    _student_progress_page_action($courseid, $userid);
    _student_progress_page_scripts();

    // Prepare data
    $mm_data_list = [];
    $get_mm_data = function($key) use (&$mm_data_list, $userid){
        return $mm_data_list[$key][$userid] ?? null;
    };
    grade_info::prepare_s($courseid, $userid);
    assign_info::prepare_submission_info_records($courseid, $userid);
    $sm_config = get_site_and_course_block_config($courseid, SM);
    $tt_config = get_site_and_course_block_config($courseid, TT);
    $block_config = is_aTT ? $tt_config : $sm_config;
    $tt_config->resubmission_assignments =
        ($tt_config->enableresubmissions && isset($tt_config->resubmission_assignments)) ? $tt_config->resubmission_assignments : [];
    $rm_config = get_rm_config($tt_config);
    $deadlinemanagerinstalled = !empty(\core_plugin_manager::instance()->get_plugin_info(is_aTT ? PLUGIN_NAME : PLUGIN_TT));
    $kica = get_kica_if_exists($courseid);
    $participation_power = $summative_score = $formative_score = $divided_score = $submissions_count = $max_course_attempts = null;
    $pp_status = '';
    $rm_check_pp = ($rm_config && $rm_config->check_pp);
    $show_pp = $tt_config->participationpower_option ?? false;

    $table = new \html_table();
    $table->id = 'datatable';
    $table->attributes = ['class' => 'activity-table student-progress'];

    $MM = MM::get_MM_by_params(['course' => $course, 'set_students' => [$userid => $user]]);
    $filter = [MM::ST_ALL, MM::BY_ACTIVITY, MM::BY_USER, MM::GET_TAGS, MM::USE_DEADLINE, MM::USE_KICA];
    $mm_data_list = $MM->get_data($filter, false, MM::MOD_ALL);
    $activities = get_important_activities_by_compare_list($courseid, $userid, $mm_data_list);

    // user activity status
    $activities_info = new activity_status();
    if (!empty($activities)) {
        $activities_info->check_activities_list($activities, $userid, false, $kica, $mm_data_list);
        switch($fs_type){
            case FORMATIVE_ACTIVITY:
                $current_info = $activities_info->get_formative_activity_status();
                break;
            case SUMMATIVE_ACTIVITY:
                $current_info = $activities_info->get_summative_activity_status();
                break;
            default:
                $current_info = $activities_info;
        }

        if ($rm_check_pp || $show_pp){
            list($summative_score, $formative_score, $participation_power, $divided_score, $text_status) = $activities_info->get_full_pp_data();
            $pp_status = 'pp-' . $text_status;
        }
    } else {
        $current_info = $activities_info;
    }

    // Create menu section & options which depend on $rm_config

    $is_view_grade = true;

    $course_grade = get_course_grade($courseid, $userid);
    $general_info = '';
    if (is_aTT) {
        $plugincfg = get_config(PLUGIN_NAME);
        /** @noinspection PhpUnnecessaryFullyQualifiedNameInspection */
        $course_users_status = new \block_ned_teacher_tools\course_users_status($courseid);
        $users_statuses = $course_users_status->get_users_statuses($userid, true);
        $enable_course_status_controller_on_this_course = $course_users_status->enable_on_course();
        $course_status_controller = $plugincfg->enablecoursecompletionstatuscontroller && $enable_course_status_controller_on_this_course;
        if ($course_status_controller && !is_siteadmin()) {
            if (array_key_exists($userid, $users_statuses) && !empty($users_statuses[$userid]) && !empty($users_statuses[$userid]->status)) {
                /** @noinspection PhpUnnecessaryFullyQualifiedNameInspection */
                $is_view_grade = \block_ned_teacher_tools\get_check_permission_view_grade($courseid, $users_statuses[$userid]->status);
            }
        }

        $user_link = link(['/user/view.php', ['id' => $userid, 'course' => $courseid]], fullname($user), 'gi-value student');
        $general_info .= \html_writer::div(
            \html_writer::span(str('student'), 'gi-label') .
            SH::get_profile_with_menu_flag($userid, $courseid, $user_link)
        );
    }

    // Grade as link to gradebook
    if ($sm_config->replacewithkicagrade && utils::kica_gradebook_enabled($courseid)) {
        $gradebook_url = new \moodle_url('/local/kica/grade_user.php', ['courseid' => $courseid, 'group' => $groupid, 'participant' => $userid]);
    } else {
        $gradebook_url = new \moodle_url('/grade/report/user/index.php', ['id' => $courseid, 'group' => $groupid, 'userid' => $userid]);
    }
    if ($is_view_grade) {
        $general_info .= \html_writer::div(
            link($gradebook_url,
                \html_writer::span(str('currentgrade'), 'gi-label') .
                \html_writer::span($course_grade, 'gi-value'), '', ['title' => str('opensmth', str('gradebook'))]
            )
        );
    } else {
        $general_info .= \html_writer::div(\html_writer::span(str('currentgrade'), 'gi-label') .
            \html_writer::span(fa('tt-status-icon tt-si-waitingfinalgrade'), 'gi-value'), 'review-modal-message-text'
        );
    }

    if ($rm_config || $show_pp){
        if ($rm_config){
            $submissions_count = assign_info::get_course_submissions_count($courseid, $userid, $tt_config);
            $rm_config = update_rm_config($rm_config, $submissions_count, $participation_power);
            $table->attributes['data-resubmit_max_time'] = $rm_config->max_time;
        }

        if (($rm_config->show_pp ?? false) || $show_pp){
            if ($summative_score){
                $pp_value = \html_writer::span( \html_writer::tag('i',' ', ['class' => 'fa fa-rocket']) .
                        \html_writer::span(str($pp_status), 'pp-status'),
                        "gi-value participationpower $pp_status", ['title' => $participation_power]) .
                    \html_writer::link('#', ' ',
                        [
                            'class' => 'participationpower-help',
                            'data-data' => "$summative_score $formative_score $divided_score $participation_power $pp_status",
                            'data-course' => $course->shortname
                        ]
                    );
            } else {
                $pp_value = '-';
            }
            $general_info .= \html_writer::div(\html_writer::span(SH::$C::str('participationpower'), 'gi-label') . $pp_value);
        }

        if ($rm_config){
            $max_course_attempts = $rm_config->max_course_attempts ? $rm_config->max_course_attempts : '∞';
            $general_info .= \html_writer::div(
                \html_writer::span(str('resubmissionused'), 'gi-label') .
                \html_writer::span($submissions_count . '/' . $max_course_attempts, 'gi-value')
            );
        }
    }

    $tag_options = [];
    if (is_aTT && ($tt_config->showtagsfilter ?? true)){
        $context = \context_course::instance($courseid);
        $contexts = $context->get_child_contexts();
        $contexts[] = $context;
        $tags = \core_tag_tag::get_tags_by_area_in_contexts('core', 'course_modules', $contexts);
        foreach ($tags as $_tag){
            $tag_options[$_tag->id] = $_tag->get_display_name(true);
        }
        $studentselector .= tags_print_course_menu(new \moodle_url($base_url, $base_url_param), $tag_options, $tag);
    }

    $menu_section = function($content, $title='', $title_link=false, $add_class=''){
        $add_class = $add_class ? $add_class : $title;
        if ($title_link){
            $title = $title ? link($title_link, str($title), 'sp-title') : '';
        } else {
            $title = $title ? \html_writer::div(str($title), 'sp-title') : '';
        }
        return \html_writer::div($title . $content, 'sp-section ' . $add_class);
    };
    $sp_menu = '';
    if ($block_config->showgeneralinfosection ?? true) {
        $sp_menu .= $menu_section($general_info, 'generalinfo');
    }
    $url_params = $base_url_param;
    $url_params['show'] = activity_status::STATUS_ALL;

    if ($tt_config->participationpower_option ?? 0){
        if ($block_config->showsummativeactivitiessection ?? true) {
            $url_params['fs_type'] = SUMMATIVE_ACTIVITY;
            $act_menu = get_activity_menu($activities_info->get_summative_list(), $base_url, $url_params);
            $sp_menu .= $menu_section($act_menu, 'summativeactivities', [$base_url, $url_params]);
        }
        if ($block_config->showformativeactivitiessection ?? true) {
            $url_params['fs_type'] = FORMATIVE_ACTIVITY;
            $act_menu = get_activity_menu($activities_info->get_formative_list(), $base_url, $url_params);
            $sp_menu .= $menu_section($act_menu, 'formativeactivities', [$base_url, $url_params]);
        }
    } else {
        $url_params['fs_type'] = USUAL_ACTIVITY;
        $act_menu =  get_activity_menu($activities_info->get_list(), $base_url, $url_params);
        $sp_menu .= $menu_section($act_menu, 'studentprogress', [$base_url, $url_params]);
    }

    // Badges

    $table_badges = '';
    $badge = function($label, $url_params, $add_class='', $attr=[]) use ($base_url){
        $attr['title'] = isset($attr['title']) ? $attr['title'] : str('removefilter');
        return link([$base_url, $url_params], $label, 'ned-badge ' . $add_class, $attr);
    };

    if ($show !== activity_status::STATUS_ALL){
        $table_badges .= $badge($show, array_merge($base_url_param, ['show' => 'all']));
    }

    if ($fs_type){
        $label = $fs_type == SUMMATIVE_ACTIVITY ? 'summativeactivities' : 'formativeactivities';
        $table_badges .= $badge($label, array_merge($base_url_param, ['fs_type' => USUAL_ACTIVITY]));
    }

    if ($rm_config){
        $max_time = $tt_config->daysforresubmission ? $tt_config->daysforresubmission * 24 : '∞';
        $table_badges .= SH::link(['#'], 'resubmission_modal:link', 'resubmission-help',
            ['data-data' => "$max_time $rm_config->show_pp $participation_power $submissions_count $max_course_attempts"]);
    }

    // Fill the main table

    if(!empty($activities)){
        $table->head = [cell(str('activityname'), 'activity-name')];

        if ($block_config->showduedatecolumn ?? true){
            $table->head[] = cell(str('duedate'), 'activity-type');
        }
        $table->head[] = cell(str('submitted'), 'activity-submitted');
        $table->head[] = cell(str('marked'), 'activity-graded');
        $table->head[] = cell(str('yourgrade'), 'activity-grade');

        $tag_exist = $tag < 1 || isset($tag_options[$tag]);
        foreach ($activities as $activity) {
            $status = $current_info->get_status($activity->id);
            if (!$current_info::is_this_status($status, $show)) {
                continue;
            }

            $tags = $current_info->get_tags($activity->id);
            if (is_aTT){
                if($tag != 0 && $tag_exist && !($tag == -1 && empty($tags)) &&
                    !$current_info->check_tag_by_name($activity->id, $tag_options[$tag] ?? '')){
                    continue;
                }
            }

            $table->data = array_merge($table->data,
                render_activity_row($activity, $userid, (bool)$kica, $deadlinemanagerinstalled,
                    $rm_config, $status, $tags, $get_mm_data($activity->id)));
        }
    }

    // Put together page content
    if (is_aTT){
        $result .= \html_writer::div($studentselector, 'student-selector-wrapper');
    }
    $result .= \html_writer::div($sp_menu, 'sp-menu') . \html_writer::div($table_badges, 'ned-table-badges');

    if (!empty($table->data)){
        $result .= \html_writer::table($table);
    } else {
        if ($show != activity_status::STATUS_ALL || $fs_type){
            $result .= \html_writer::div(str('noactivitybyfilters'), 'no-activity');
        } else {
            $result .= \html_writer::div(str('noactivity'), 'no-activity');
        }
    }

    if ($rm_config){
        $result .= \html_writer::div(
            SH::$C::render_from_template('resubmission_legend',
                ['title' => str('iconlegend'), 'classname' => 'description']),
            'resubmission-legend');
    }

    // Render page
    $print_result($result);
}


/**
 * Load scripts for the student_progress page
 */
function _student_progress_page_scripts(){
    global $PAGE;

    SH::$C::js_call_amd('add_sorter', 'add_sort', ['#datatable']);
    SH::$C::js_call_amd('student_progress', 'init', [$PAGE->context->id]);
    SH::$C::js_call_amd('add_timer', 'add', ['#mark-interface table .timer[data-datetime]']);
    SH::$C::js_call_amd('participationpower', 'init', ['#mark-interface a.participationpower-help']);
    if (!is_aTT){
        return;
    }

    SH::js_call_amd('reviewmodalmessage', 'init');

    // notification template for remove submission feature
    $note = new \local_ned_controller\support\ned_notify_simple();
    $note->set_type($note::TYPE_NOTICE);
    $note->cancel_button = true;
    $note->ok_text = get_string('continue');
    SH::js_call_amd('student_progress_menu', 'init', [$note->js_export()]);
}

/**
 * Check actions for the student_progress page
 * Note: return void, but can redirect page
 *
 * @param $courseid
 * @param $userid
 *
 */
function _student_progress_page_action($courseid, $userid){
    global $PAGE, $COURSE;
    if (!is_aTT){
        return;
    }

    $action = optional_param('action', '', PARAM_TEXT);
    $action_cmid = optional_param('cmid', 0, PARAM_INT);
    if (empty($action) || empty($action_cmid) || empty($userid) || !in_array($action, SP_ACTIONS)){
        return;
    }

    $url = clone($PAGE->url);
    $url->remove_params(['action', 'cmid']);
    $course_context = \context_course::instance($COURSE->id);
    if (!SH::has_capability($action, $course_context)){
        redirect($url, SH::str('actiondenied'), null, \core\notification::ERROR);
        return;
    }

    $ned_assign = SH::get_assign_by_cm($action_cmid);
    if (!$ned_assign->is_valid($courseid)){
        return;
    }

    $cm_name = $ned_assign->get_course_module()->name;
    $success = null;
    $messages = [
        SP_ACTION_REMOVESUBMISSION => [
            true => 'submissionremoved',
            false => 'submissioncantremoved',
        ],
        SP_ACTION_ALLOWRESUBMISSION => [
            true => 'resubmissionadded',
            false => 'resubmissioncantadded',
        ],
    ];
    $message = null;

    switch ($action){
        case SP_ACTION_REMOVESUBMISSION:
            $success = $ned_assign->remove_last_assign_submission($userid);
            break;
        case SP_ACTION_ALLOWRESUBMISSION:
            $due_days = optional_param('duedays', 0, PARAM_INT);
            $not_credit = optional_param('notcredit', false, PARAM_BOOL);
            $set_duedate = $due_days ? (time() + $due_days*DAYSECS) : 0;
            $success = $ned_assign->force_add_attempt($userid, $not_credit ? null : $userid, $set_duedate);
            break;
    }

    if (is_null($success)){
        return;
    }

    $success = (bool)$success;
    if (empty($message)){
        $message = $messages[$action][$success] ?? '';
        if (empty($message)){
            return;
        }

        $message = str($message, $cm_name);
    }

    $message_type = $success ? \core\notification::SUCCESS : \core\notification::WARNING;
    redirect($url, $message, null, $message_type);
}

/**
 * Sort activities in moodle gradebook order (by \grade_item->sortorder)
 * @param \cm_info[] $activities
 *
 * @return \cm_info[] - [\grade_item->sortorder => $mod]
 */
function sort_activities_as_gradebook($activities){
    foreach ($activities as $mod){
        if (!isset($courseid)){
            $courseid = $mod->get_course()->id;
        }
        $grade_item = get_grade_item($courseid, $mod->modname, $mod->instance);
        if ($grade_item){
            $activities_by_sortorder[$grade_item->sortorder] = $mod;
        }
    }
    ksort($activities_by_sortorder);
    return $activities_by_sortorder;
}


// Small functions, which often used locally

/**
 * Moodle get_string for this plugin
 *
 * @param      $identifier
 * @param null $params
 * @param null $plugin
 *
 * @return string
 */
function str($identifier, $params=null, $plugin=null){
    return SH::str($identifier, $params, $plugin);
}

/**
 * Moodle get_string for this plugin, but with check of existing such string
 *
 * @param array|string $identifier_params - [$identifier, $params, $plugin] for str(), or only $identifier as string
 * @param null  $def - if there are no $identifier, use $def
 * @param bool  $use_def_str - return str($def) as default if true, just $def otherwise
 *
 * @return string
 */
function str_check($identifier_params=[], $def=null, $use_def_str=true){
    static $string_manager = null;
    $identifier_params_def = ['', null, null];
    if (!$string_manager){
        $string_manager = get_string_manager();
    }

    if (is_array($identifier_params)){
        $identifier_params = $identifier_params + $identifier_params_def;
    } else {
        $identifier_params = [$identifier_params] + $identifier_params_def;
    }
    list($identifier, $params, $plugin) = $identifier_params;
    return SH::str_check($identifier, $params, $def, $plugin);
}

/**
 * @param           $check
 * @param int|null  $data
 * @param string    $format - see format of strftime() function
 *
 * @return string
 */
function ned_date($check, $data=null, $format = '%d %b %Y @ %l:%M%p'){
    if (!$check){
        return '-';
    }
    $data = is_null($data) ? $check : $data;
    if (strpos($format, '%p') !== false){
        $format = str_replace('%p', strtoupper(userdate($data, '%p')), $format);
    }
    return userdate($data, $format);
}

/**
 * @param string|\moodle_url|array  $url_params - if it's array, that used [$url_text='', $params=null, $anchor=null]
 * @param string $text
 * @param string|array $class
 * @param array  $attr
 *
 * @return string
 * @throws \moodle_exception
 */
function link($url_params='', $text='', $class=null, $attr=[]){
    if ($url_params instanceof \moodle_url) {
        $m_url = $url_params;
    } else {
        if (is_string($url_params)){
            list($t_url, $params, $anchor) = [$url_params, null, null];
        } else {
            list($t_url, $params, $anchor) = $url_params + ['', null, null];
        }

        $m_url = new \moodle_url($t_url, $params, $anchor);
    }

    $attr['class'] = (isset($attr['class'])) ? $attr['class'] : '';
    if (is_array($class)){
        $class = join(' ', $class);
    }
    if (is_null($class) && strpos($text, ' ') === false){
        $class = $text;
    }
    $attr['class'].= $class ?: '';

    if (!empty($text) and get_string_manager()->string_exists($text, PLUGIN_NAME)) {
        $text = str($text);
    }

    return \html_writer::link($m_url, $text, $attr);
}

/**
 * @param string $filename
 * @param string $class
 * @param string $plugin
 * @param array  $attr
 *
 * @return string
 */
function img($filename, $class='icon', $plugin='moodle', $attr=[]){
    global $OUTPUT;
    $plugin = is_null($plugin) ? PLUGIN_NAME : $plugin;
    $url = '/blocks/' . PLUGIN . '/pix/' . $filename;
    $url = file_exists(DIRROOT . $url) ? $url :
        (file_exists(DIRROOT . $filename) ? $filename : $OUTPUT->image_url($filename, $plugin));
    $attr['class'] = (isset($attr['class'])) ? $attr['class'] : '';
    $attr['class'].= $class;
    $alt = isset2($attr, ['alt'], isset2($attr, ['title'], ''));
    return \html_writer::img(new \moodle_url($url), $alt, $attr);
}

/**
 * Return image by activity status, empty string for unknowing status
 *
 * @param string $status
 * @param string $class
 * @param bool   $notattempte_is_ungraded
 *
 * @return string
 */
function status_img($status='', $class='icon', $notattempte_is_ungraded=false){
    $img = function($filename, $title=null) use ($class, $status){
        $title = $title ?? $status;
        return img($filename, $class, 'moodle', ['title' => str($title)]);
    };
    $fa = function($class, $title=null) use ($status){
        $title = $title ?? $status;
        return fa($class, '', str($title));
    };

    switch($status){
        case 'graded_kica':
        case 'graded_kica_summative':
            return $fa('fa-check-square');
        case 'completed' :
        case 'graded_kica_formative':
        case 'marked':
            return $fa('fa-check-square-o');
        case 'saved':
        case 'draft':
            return $fa('fa-floppy-o');
        case 'waitingforgrade':
        case 'submitted':
        case 'unmarked':
        case 'ungraded_kica_formative':
            return $fa('fa-clock-o');
        case 'ungraded_kica':
        case 'ungraded_kica_summative':
            return $fa('fa-clock-o2');
        case 'incompleted':
        case 'incompleted_formative':
            return $fa('fa-window-close-o');
        case 'incompleted_summative':
            return $fa('fa-window-close');
        case 'notattempted':
        case 'unsubmitted':
        case 'not_submitted':
            return $img($notattempte_is_ungraded ? 'ungraded.gif' : 'not_submitted.gif', 'unsubmitted');
        case 'kica_zerograde':
            return \html_writer::span(0, 'kica-zero', ['title' => str($status)]);
        case 'excluded':
            return $fa('fa-first-order');
        case 'deadline-past':
            return $fa('deadline-past');
        case 'passed':
        case 'graded':
            return '';
    }
    return '';
}

/**
 * @param string $class
 *
 * @param string $content
 * @param string $title
 *
 * @return string
 */
function fa($class='', $content='', $title=''){
    $attr = ['class' => 'icon fa ' . $class, 'aria-hidden' => 'true'];
    if (!empty($title)){
        $attr['title'] = $title;
    }
    return \html_writer::tag('i', $content, $attr);
}

/**
 * @param \cm_info|\stdClass $activity
 * @param int                $icon_size
 * @param bool               $only_icon
 * @param string             $add_class
 * @param array              $add_params
 *
 * @return string
 */
function mod_link($activity, $icon_size=20, $only_icon=false, $add_class='', $add_params=[]){
    $mod_icon = img('icon', 'mod-icon', $activity->modname, ['height' => $icon_size, 'width' => $icon_size]);
    $mod_text = $only_icon ? $mod_icon : $mod_icon .' '. $activity->name;
    $add_params['title'] = $activity->name;
    return link(['/mod/' . $activity->modname . '/view.php', ['id' => $activity->id]],
        $mod_text, 'mod-link ' . $add_class, $add_params);
}

/**
 * @param null   $cells
 * @param string $class
 * @param null   $attr
 *
 * @return \html_table_row
 */
function row($cells=null, $class='', $attr=null){
    if (!is_array($cells) && !is_null($cells)){
        $cells = [$cells];
    }
    $row = new \html_table_row($cells);
    $row->attributes['class'] = $class;
    if ($attr){
        $row->attributes = array_merge($row->attributes, $attr);
    }
    return $row;
}

/**
 * @param null   $text
 * @param string|array $class
 * @param null   $attr
 *
 * @return \html_table_cell
 */
function cell($text=null, $class='', $attr=null){
    $cell = new \html_table_cell($text);
    if (is_array($class)){
        $class = join(' ', $class);
    }
    $cell->attributes['class'] = $class;
    if ($attr){
        $cell->attributes = array_merge($cell->attributes, $attr);
    }
    return $cell;
}

/**
 * @param        $url
 * @param string $message
 * @param null   $delay
 * @param string $messagetype
 *
 */
function redirect($url, $message='', $delay=null, $messagetype = \core\output\notification::NOTIFY_INFO){
    global $PAGE;
    if ($PAGE->state == \moodle_page::STATE_PRINTING_HEADER){
        $PAGE->set_state(\moodle_page::STATE_IN_BODY);
    }
    \redirect($url, $message, $delay, $messagetype);
    die;
}

/**
 * Print notification and link "continue" as redirect attempt
 *
 * @param        $url
 * @param string $message
 * @param string $messagetype
 * @param bool   $return
 *
 * @return string
 */
function redirect_continue($url, $message='', $messagetype = \core\output\notification::NOTIFY_INFO, $return=false){
    global $OUTPUT;
    $output = '';
    if (!empty($message)){
        $output .= $OUTPUT->notification($message, $messagetype);
    }
    $output .= \html_writer::div('(' . \html_writer::link($url, get_string('continue')) . ')', 'continuebutton');

    if (!$return){
        echo $output;
    }
    return $output;
}

/**
 * @param null $made_by_rm - true if only made_by_rm, false if only not_made_by_rm
 * @param null $courseid
 *
 * @return array [$courses_by_cat[], $records_by_cat[]]
 */
function get_unused_resubmissions($made_by_rm=null, $courseid=null){
    global $DB;

    $additional_g_g_h_condition = '';
    if (!is_null($made_by_rm)){
        if ($made_by_rm){
            $additional_g_g_h_condition = "AND g_g_h.loggeduser = a_s.userid";
        } else {
            $additional_g_g_h_condition = "AND g_g_h.loggeduser <> a_s.userid";
        }
    }

    $additional_where = '';
    if ($courseid){
        $additional_where .= "AND c.id = $courseid";
    }

    $sql = "SELECT g_g_h.id, 
        CONCAT(u.firstname, ' ', u.lastname) AS 'username',
        c.fullname AS 'course',
        a.name AS 'assign',
        
        u.id AS 'userid',
        c.id AS 'courseid',
        a.id AS 'activityid',
        cm.id AS 'cmid',
        g_g_h.loggeduser = u.id AS 'made_by_rm',
        a_s.timecreated,
        
        a_s.attemptnumber AS 'Attempt'
        FROM {assign_submission} AS a_s
        LEFT JOIN {assign} AS a
            ON a_s.assignment = a.id
        LEFT JOIN {grade_items} AS g_i
            ON a_s.assignment = g_i.iteminstance
            AND g_i.itemtype = \"mod\"
            AND g_i.itemmodule = \"assign\"
            AND g_i.courseid = a.course
        LEFT JOIN {assign_grades} a_g
            ON a_s.assignment = a_g.assignment
            AND a_g.userid = a_s.userid
            AND a_g.attemptnumber = a_s.attemptnumber
        LEFT JOIN {grade_grades_history} AS g_g_h
            ON g_i.id = g_g_h.itemid
            AND g_g_h.timemodified = a_s.timecreated
            AND g_g_h.source = \"mod/assign\"
            AND g_g_h.userid = a_s.userid
            AND g_g_h.usermodified = a_s.userid
            $additional_g_g_h_condition
        LEFT JOIN {course} AS c 
            ON c.id = a.course
        LEFT JOIN {user} AS u
            ON u.id = a_s.userid
        LEFT JOIN {context} AS cntxt
            ON (cntxt.contextlevel = 50 
            AND cntxt.instanceid = c.id)
        LEFT JOIN {role_assignments} AS ra
            ON ra.userid = u.id 
            AND ra.contextid = cntxt.id
        LEFT JOIN {role_assignments} AS ra2
            ON ra2.userid = u.id 
            AND ra2.contextid = cntxt.id
            AND ra2.id > ra.id
        LEFT JOIN {role} AS r 
            ON r.id = ra.roleid
        
        LEFT JOIN {course_modules} AS cm
            ON cm.course = c.id
            AND cm.instance = a.id
        
        WHERE a_s.status = 'reopened' 
            AND a_s.latest = '1' 
            AND r.shortname = 'student' 
            AND ra2.id IS NULL
            AND g_g_h.id IS NOT NULL 
            AND (a_g.timemodified IS NULL OR a_g.grade IS NULL OR a_g.grade <= 0)
            $additional_where
        ORDER BY c.id, g_g_h.id
    ";
    $records = $DB->get_records_sql($sql);
    $courses = [];
    $records_by_cat = [];
    $courses_by_cat = [];
    $unic = [];
    foreach (RESUBMISSIONS_KEYS as $KEY){
        $records_by_cat[$KEY] = [];
        $courses_by_cat[$KEY] = [];
    }
    foreach ($records as $record){
        if (isset($unic[$record->userid][$record->cmid])){
            continue;
        }
        $unic[$record->userid][$record->cmid] = true;

        if (!isset($courses[$record->courseid])){
            $rm_config = get_rm_config_by_courseid($record->courseid);
            if (!$rm_config){
                $courses[$record->courseid] = false;
            } else {
                $courses[$record->courseid] = $rm_config->assignments;
            }
        }

        $record->controlled_rm = $courses[$record->courseid] && isset($courses[$record->courseid][$record->cmid]) &&
            $courses[$record->courseid][$record->cmid];
        if ($record->made_by_rm){
            $records_by_cat[RESUBMISSIONS_MADE_BY_RM][$record->id] = $record;
            $courses_by_cat[RESUBMISSIONS_MADE_BY_RM][$record->courseid] = $record->course;
            if ($record->controlled_rm){
                $records_by_cat[RESUBMISSIONS_MADE_BY_RM_CONTROLLED_BY_RM][$record->id] = $record;
                $courses_by_cat[RESUBMISSIONS_MADE_BY_RM_CONTROLLED_BY_RM][$record->courseid] = $record->course;
            } else {
                $records_by_cat[RESUBMISSIONS_MADE_BY_RM_CONTROLLED_NOT_BY_RM][$record->id] = $record;
                $courses_by_cat[RESUBMISSIONS_MADE_BY_RM_CONTROLLED_NOT_BY_RM][$record->courseid] = $record->course;
            }
        } else {
            $records_by_cat[RESUBMISSIONS_MADE_NOT_BY_RM][$record->id] = $record;
            $courses_by_cat[RESUBMISSIONS_MADE_NOT_BY_RM][$record->courseid] = $record->course;
            if ($record->controlled_rm){
                $records_by_cat[RESUBMISSIONS_MADE_NOT_BY_RM_CONTROLLED_BY_RM][$record->id] = $record;
                $courses_by_cat[RESUBMISSIONS_MADE_NOT_BY_RM_CONTROLLED_BY_RM][$record->courseid] = $record->course;
            } else {
                $records_by_cat[RESUBMISSIONS_MADE_NOT_BY_RM_CONTROLLED_NOT_BY_RM][$record->id] = $record;
                $courses_by_cat[RESUBMISSIONS_MADE_NOT_BY_RM_CONTROLLED_NOT_BY_RM][$record->courseid] = $record->course;
            }
        }


        $records_by_cat[RESUBMISSIONS_ALL][$record->id] = $record;
        $courses_by_cat[RESUBMISSIONS_ALL][$record->courseid] = $record->course;
    }

    return [$courses_by_cat, $records_by_cat];
}

/**
 * Remove last submission and try to revert grade to previous state
 *  return false if can do nothing, true if can at least change assign_submission
 *
 * WARNING: There are none security check
 *
 * @param \cm_info|int|string $cmid
 * @param int|string          $userid
 * @param null|string         $check_status - string status, if you wish check submission
 *
 * @return bool
 */
function remove_last_submission($cmid, $userid, $check_status=null){
    $ned_assign = SH::get_assign_by_cm($cmid);
    return $ned_assign->remove_last_assign_submission($userid, $check_status);
}

/**
 * Return html checkbox as checkbox & link (the same to single selector)
 *
 * @param             $value
 * @param \moodle_url $url
 * @param string      $name
 * @param string      $text
 * @param string      $class
 * @param null        $attributes
 *
 * @return string
 */
function single_checkbox($value, \moodle_url $url, $name, $text='', $class='', $attributes=null){
    $url = new \moodle_url($url);
    $url->param($name, (int)(!$value));
    $checkbox =  \html_writer::checkbox($name, $value, $value, $text,
        ['onclick' => "window.location.href = '{$url->out(false)}';"]);
    return \html_writer::div($checkbox, $class, $attributes);
}

/**
 * Return string selector if $render=true, \single_select otherwise
 *
 * @param        $url
 * @param        $name
 * @param array  $options
 * @param string $selected
 * @param null   $label
 * @param null   $formid
 * @param null   $nothing
 * @param array  $attributes
 * @param bool   $render
 *
 * @return \single_select | string
 */
function single_select($url, $name, $options=[], $selected='', $label=null, $formid=null, $nothing=null, $attributes=[], $render=true){
    global $OUTPUT;
    $url = new \moodle_url($url);
    $url->remove_params($name);
    $select = new \single_select($url, $name, $options, $selected, $nothing, $formid);

    if ($label){
        $select->set_label($label);
    } elseif (array_key_exists('label', $attributes)) {
        $select->set_label($attributes['label']);
        unset($attributes['label']);
    }
    if (isset2($attributes, 'disabled')){
        $select->disabled = true;
    }
    $select->attributes = $attributes ? $attributes : [];

    return $render ? $OUTPUT->render($select) : $select;
}

function choose_students_by_group($allstudents, $groups, $groupid){
    $enrolledusers = $allstudents;
    if ($groupid == GROUP_NONE){
        $enrolledusers = [];
    } elseif($groupid != GROUP_ALL){
        $enrolledusers = isset($groups[$groupid]) ? $groups[$groupid]->users : [];
    }
    return $enrolledusers;
}

/**
 * Return grade for given user or all users.
 *
 * @global object
 * @global object
 * @param object $forum
 * @param int $userid optional user id, 0 means all users
 * @return array array of grades, false if none
 */
function forum_get_user_grades($forum, $userid = 0) {
    global $CFG;

    require_once($CFG->dirroot.'/rating/lib.php');

    $ratingoptions = new \stdClass;
    $ratingoptions->component = 'mod_forum';
    $ratingoptions->ratingarea = 'post';

    //need these to work backwards to get a context id. Is there a better way to get contextid from a module instance?
    $ratingoptions->modulename = 'forum';
    $ratingoptions->moduleid   = $forum->id;
    $ratingoptions->userid = $userid;
    $ratingoptions->aggregationmethod = $forum->assessed;
    $ratingoptions->scaleid = $forum->scale;
    $ratingoptions->itemtable = 'forum_posts';
    $ratingoptions->itemtableusercolumn = 'userid';

    $rm = new \rating_manager();
    return $rm->get_user_grades($ratingoptions);
}

/**
 * Return grade function for mod, if exist, null otherwise
 * @param \cm_info|\stdClass $mod
 *
 * @return string|null
 */
function get_grade_function($mod){
    global $CFG;
    $gradefunction = $mod->modname . "_get_user_grades";
    $local_gradefunction = '\\' . PLUGIN_NAME . '\\' . $gradefunction;
    if(function_exists($local_gradefunction)){
        return $local_gradefunction;
    }

    $libfile = "$CFG->dirroot/mod/$mod->modname/lib.php";
    if (!file_exists($libfile)){
        return null;
    }

    require_once($libfile);
    return function_exists($gradefunction) ? $gradefunction : null;
}

/**
 * Check $obj on isset by all $keys
 *  Return value, if it exists, $def otherwise
 * @param       $obj
 * @param array $keys
 * @param null  $def
 *
 * @return array|mixed|null
 */
function isset2($obj, $keys=[], $def=null){
    if (is_object($obj)){
        $obj = (array)$obj;
    } elseif(!is_array($obj)){
        return $def;
    }
    $keys = is_array($keys) ? $keys : (array)$keys;
    foreach ($keys as $key){
        if(!isset($obj[$key])){
            return $def;
        }
        $obj = $obj[$key];
    }
    return $obj;
}

/**
 * Return $key, it exists in $obj, $def otherwise
 *  if $def not set, return first key from $obj
 *  use $return_null if wish to get null as $def value
 *
 * @param      $obj
 * @param      $key
 * @param null $def
 * @param bool $return_null
 *
 * @return int|string|null
 */
function isset_key($obj, $key, $def=null, $return_null=false){
    if (is_object($obj)){
        $obj = (array)$obj;
    } elseif(!is_array($obj)){
        return $def;
    }
    if (empty($obj)){
        return $def;
    }
    reset($obj);
    $def = (is_null($def) && !$return_null) ? key($obj) : $def;
    $key = isset($obj[$key]) ? $key : $def;
    return $key;
}

/**
 * Return $val, it exists in $list, $def otherwise
 *  if $def not set, return first val from $list
 *  use $return_null if wish to get null as $def value
 *
 * @param array $list
 * @param       $val
 * @param null  $def
 * @param bool  $return_null
 *
 * @return mixed|null
 */
function isset_in_list($list, $val, $def=null, $return_null=false){
    if(!is_array($list) || empty($list)){
        return $def;
    }
    if (in_array($val, $list)){
        return $val;
    } elseif (is_null($def)){
        if ($return_null){
            return null;
        } else {
            return reset($list);
        }
    } else {
        return $def;
    }
}

/**
 * Return array with keys from options and values as str(option)
 *
 * @param        $options
 * @param array  $result
 * @param string $surround
 * @param array  $additional_data
 *
 * @return array
 */
function get_str_options($options, $result=[], $surround='', $additional_data=[]){
    $additional_data = (array)$additional_data;
    foreach ($options as $option){
        $add = isset($additional_data[$option]) ? '('.$additional_data[$option].')' : '';
        $result[$option] = $surround . str($option) . $add . $surround;
    }
    return $result;
}

/**
 * Returns list of users enrolled into course.
 *
 * @param $courseid
 * @param \context $context
 * @param string $withcapability
 * @param int $groupid 0 means ignore groups, USERSWITHOUTGROUP without any group and any other value limits the result by group id
 * @param string $userfields requested user record fields
 * @param string $orderby
 * @param int $limitfrom return a subset of records, starting at this point (optional, required if $limitnum is set).
 * @param int $limitnum return a subset comprising this many records (optional, required if $limitfrom is set).
 * @return array of user records
 */
function get_enrolled_users($courseid, \context $context, $withcapability = '', $groupid = 0, $userfields = 'u.*', $orderby = null,
    $limitfrom = 0, $limitnum = 0) {

    $onlyactive = !get_cache_show_inactive($courseid);
    return \get_enrolled_users($context, $withcapability, $groupid, $userfields, $orderby, $limitfrom, $limitnum, $onlyactive);
}

/**
 * @param $mm_data
 *
 * @return string
 */
function get_status_by_mm_data($mm_data){
    $status = 'notattempted';
    if (!$mm_data){
        return $status;
    }

    $st_mod = SH::stdClass2($mm_data);
    $get_usual_complete = function($competed='completed', $incompleted='incompleted', $notattempted='notattempted') use ($st_mod){
        if ($st_mod->shouldpass){
            if ($st_mod->activity_completion){
                return $incompleted;
            } else {
                return $st_mod->completed_successfully ? $competed : $incompleted;
            }
        } else {
            return $st_mod->activity_completion ? $notattempted : $competed;
        }
    };

    if ($st_mod->excluded){
        $status = 'excluded';

    } elseif ($st_mod->activity_completion && $st_mod->completed){
        if ($st_mod->kica_activity){
            $status = $st_mod->is_summative ? 'graded_kica_summative' : 'graded_kica_formative';
        } else {
            $status = 'completed';
        }
    } elseif ($st_mod->kica_activity){
        if ($st_mod->{MM::ST_UNMARKED}){
            if ($st_mod->is_summative ?? false)
                $status = 'ungraded_kica_summative';
            elseif ($st_mod->is_formative ?? false)
                $status = 'ungraded_kica_formative';

        } elseif ($st_mod->{MM::ST_MARKED} && $st_mod->kica_zerograde){
            $status = 'kica_zerograde';
        } elseif ($st_mod->{MM::ST_MARKED} || $st_mod->{MM::ST_SUBMITTED}){
            $status = $get_usual_complete($st_mod->is_summative ? 'graded_kica_summative' : 'graded_kica_formative',
                $st_mod->is_summative ? 'incompleted_summative' : 'incompleted_formative'
            );
        }
    } else {
        if ($st_mod->{MM::ST_UNMARKED}){
            $status = 'waitingforgrade';
        } elseif ($st_mod->{MM::ST_MARKED} || $st_mod->{MM::ST_SUBMITTED}){
            $status = $get_usual_complete();
        }
    }

    if ($status == 'notattempted' && $st_mod->{MM::ST_DRAFT}){
        $status = 'draft';
    }

    return $status;
}

/**
 * @return bool
 */
function is_kica_exists(){
    global $CFG;
    static $_res = null;

    if (is_null($_res)){
        $_res = false;
        if ($plugininfo = \core_plugin_manager::instance()->get_plugin_info('local_kica')){
            $libs = [
                '/local/kica/classes/grade.php',
                '/local/kica/classes/helper.php',
                '/local/kica/classes/kica_item.php',
                '/local/kica/lib.php',
            ];
            foreach ($libs as $lib){
                $kica_lib = $CFG->dirroot . $lib;
                $_res = file_exists($kica_lib);
                if (!$_res){
                    break;
                }
                require_once($kica_lib);
            }
        }
    }

    return $_res;
}

/**
 * @param $courseid
 *
 * @return \stdClass|bool|null
 */
function get_kica_if_exists($courseid){
    global $DB;
    static $_data = [];
    if (!is_kica_exists()){
        return false;
    }

    if (!isset($_data[$courseid])){
        $kica = $DB->get_record('local_kica', ['courseid' => $courseid]);
        if (!($kica->enabled ?? false)){
            $kica = false;
        }

        $_data[$courseid] = $kica;
    }

    return $_data[$courseid];
}

/**
 * Return sql with FROM, JOIN and WHERE for getting user_enrolments
 *  if sent $params, necessary data will be saved here
 *
 * @param string $select
 * @param array  $where
 * @param array  $params
 * @param string $rolename
 * @param int    $courseid
 * @param int    $groupid
 * @param int    $cmid
 * @param bool   $is_active
 * @param string $other_sql
 * @param string $add_join
 * @param string $prefix
 *
 * @return string
 */
function get_sql_user_enrolments($select='ue.id', $where=[], &$params=[], $rolename=ROLE_STUDENT,
    $courseid=0, $groupid=0, $cmid=0, $is_active=true, $other_sql='', $add_join='', $prefix='ue_'){
    if (!isset($params[$prefix.'rolename'])){
        $params[$prefix.'rolename'] = $rolename;
    }

    $limit_cm = '';
    if ($cmid){
        if (!isset($params[$prefix.'cmid'])){
            $params[$prefix.'cmid'] = $cmid;
        }
        $limit_cm = "AND cm.id = :{$prefix}cmid";
    }
    if ($courseid){
        $params[$prefix.'courseid'] = $courseid;
        $where[] = "e.courseid = :{$prefix}courseid";
    }
    if ($groupid > 0){
        $params[$prefix.'groupid'] = $groupid;
        $where[] = "gr.id = :{$prefix}groupid";
    }
    $params[$prefix.'ctx_course'] = CONTEXT_COURSE;
    $params[$prefix.'ctx_module'] = CONTEXT_MODULE;
    $where[] = "(r.id = role_course.id OR r.id = role_cm.id)";

    if ($is_active){
        $where[] = "(
                ue.status = :{$prefix}active AND e.status = :{$prefix}enabled AND 
                ue.timestart <= :{$prefix}now1 AND (ue.timeend = 0 OR ue.timeend > :{$prefix}now2) AND
                u.suspended = 0 AND u.deleted = 0
            )";
        $t =  time();
        $params[$prefix.'now1'] = $t;
        $params[$prefix.'now2'] = $t;
        $params[$prefix.'enabled'] = ENROL_INSTANCE_ENABLED;
        $params[$prefix.'active'] = ENROL_USER_ACTIVE;
    }

    if (is_array($add_join)){
        $add_join = join("\n", $add_join);
    }

    $sql = "SELECT $select
            FROM {user_enrolments} ue
            JOIN {user} u
                ON u.id = ue.userid
            JOIN {enrol} e
                ON e.id = ue.enrolid
            JOIN {role} r 
                ON r.shortname = :{$prefix}rolename
                
            LEFT JOIN {context} ctx_course
                ON ctx_course.contextlevel = :{$prefix}ctx_course
                AND ctx_course.instanceid = e.courseid
            LEFT JOIN {role_assignments} ra_course
                ON ra_course.contextid = ctx_course.id
                AND ra_course.userid = ue.userid
            LEFT JOIN {role} role_course
                ON role_course.id = ra_course.roleid
                
            LEFT JOIN {course_modules} cm
                ON cm.course = e.courseid
                $limit_cm
            LEFT JOIN {context} ctx_cm
                ON ctx_cm.contextlevel = :{$prefix}ctx_module
                AND ctx_cm.instanceid = cm.id
            LEFT JOIN {role_assignments} ra_cm
                ON ra_cm.contextid = ctx_cm.id
                AND ra_cm.userid = ue.userid
            LEFT JOIN {role} role_cm
                ON role_cm.id = ra_cm.roleid
                
            LEFT JOIN (
                    SELECT grp.id, g_m.userid, grp.courseid
                    FROM {groups} grp
                    JOIN {groups_members} g_m
                       ON g_m.groupid = grp.id
            ) gr 
                ON gr.courseid = e.courseid
                AND gr.userid = ue.userid
                
            $add_join
        ";
    $where = !empty($where) ? ("\nWHERE (" . join(') AND (', $where) . ')') : '';
    return $sql.$where.$other_sql;
}

/**
 * Return course_modules_completion by WHERE conditions from options
 * if $return_first - return only one record or null
 * if only 'courseid' in $options, return array by cmid and userid of records (or array of true, if $check_complete is true)
 *
 * @param array $options
 * @param bool  $return_first
 * @param bool  $check_complete - get only records with complete status
 *
 * @return array|\stdClass|bool|null
 */
function get_course_modules_completion($options=[], $return_first=false, $check_complete=false){
    global $DB;
    $params_options = ['courseid' => 'cm.course', 'cmid' => 'cm.id', 'userid' => 'cmc.userid'];
    $params = [];
    $where = [];
    foreach ($params_options as $param => $wh){
        if (isset($options[$param])){
            $where[] = "$wh = :$param";
            $params[$param] = $options[$param];
        }
    }

    if (empty($where)){
        \print_error('error', 'error', '', "You can't call get_course_modules_completion functions without options!");
    }

    if ($check_complete){
        $where[] = "(cmc.completionstate = :completionpass OR cmc.completionstate = :completioncomplete)";
        $params['completionpass'] = COMPLETION_COMPLETE_PASS;
        $params['completioncomplete'] = COMPLETION_COMPLETE;
    }

    $where = empty($where) ? '' : ("\nWHERE (" . join(') AND (', $where) . ')');
    $sql = "SELECT cmc.*
        FROM {course_modules} cm
        JOIN {course_modules_completion} cmc ON cm.id=cmc.coursemoduleid
        $where
    ";
    $records = $DB->get_records_sql($sql, $params);
    if ($return_first){
        $records = empty($records) ? null : reset($records);
        if ($check_complete){
            $records = (bool)$records;
        }
        return $records;
    }

    if (isset($options['courseid']) && !(isset($options['cmid']) || isset($options['userid']))){
        $res = [];
        foreach ($records as $record){
            $res[$record->coursemoduleid][$record->userid] = $check_complete ? true : $record;
        }
        return $res;
    } elseif ((isset($options['cmid']) && !isset($options['userid'])) || (isset($options['userid']) && !isset($options['cmid']))){
        $res = [];
        $key = isset($options['cmid']) ? 'userid' : 'coursemoduleid';
        foreach ($records as $record){
            $res[$record->$key] = $check_complete ? true : $record;
        }
        return $res;
    }

    return $records;
}

/**
 * Print error in the moodle style
 *
 * @param      $message
 * @param      $moreinfourl
 * @param      $link
 * @param      $backtrace
 * @param null $debuginfo
 * @param bool $forcibly
 *
 * @return string
 */
function fatal_error($message, $moreinfourl, $link, $backtrace, $debuginfo = null, $forcibly=false) {
    global $CFG, $OUTPUT;

    $output = '';
    $obbuffer = '';

    if (is_object($message)){
        $info = $message;
        list($message, $moreinfourl, $link, $backtrace, $debuginfo, $errorcode) =
            [$info->message, $info->moreinfourl, $info->link, $info->backtrace, $info->debuginfo, $info->errorcode];
    }

    $msgs = explode("\n", $message);
    $message = '';
    foreach ($msgs as $msg){
        $message .= '<p class="errormessage">' . s($msg) . '</p>';
    }
    $message .= '<p class="errorcode"><a href="' . s($moreinfourl) . '">' . get_string('moreinformation') . '</a></p>';
    if (empty($CFG->rolesactive)) {
        $message .= '<p class="errormessage">' . get_string('installproblem', 'error') . '</p>';
        //It is usually not possible to recover from errors triggered during installation, you may need to create a new database or use a different database prefix for new installation.
    }
    $output .= $OUTPUT->box($message, 'errorbox alert alert-danger', null, array('data-rel' => 'fatalerror'));

    if ($CFG->debugdeveloper || $forcibly) {
        $labelsep = get_string('labelsep', 'langconfig');
        if (!empty($debuginfo)) {
            $debuginfo = s($debuginfo); // removes all nasty JS
            $debuginfo = str_replace("\n", '<br />', $debuginfo); // keep newlines
            $label = get_string('debuginfo', 'debug') . $labelsep;
            $output .= $OUTPUT->notification("<strong>$label</strong> " . $debuginfo, 'notifytiny');
        }
        if (!empty($backtrace)) {
            $label = get_string('stacktrace', 'debug') . $labelsep;
            $output .= $OUTPUT->notification("<strong>$label</strong> " . format_backtrace($backtrace), 'notifytiny');
        }
        if ($obbuffer !== '' ) {
            $label = get_string('outputbuffer', 'debug') . $labelsep;
            $output .= $OUTPUT->notification("<strong>$label</strong> " . s($obbuffer), 'notifytiny');
        }
    }

    if (empty($CFG->rolesactive)) {
        // continue does not make much sense if moodle is not installed yet because error is most probably not recoverable
    } else if (!empty($link)) {
        $output .= $OUTPUT->continue_button($link);
    }

    return $output;
}

/**
 * Print error in the moodle style from an exception
 *
 * @param \Exception|\Throwable $ex
 * @param array|string          $messages
 * @param bool                  $forcibly
 * @param null                  $a
 * @param null                  $debuginfo
 *
 * @return string
 *
 * @throws \moodle_exception
 */
function print_error($ex, $messages=[], $forcibly=true, $a=null, $debuginfo=null){
    if (is_string($ex)){
        \print_error($errorcode=$ex, $module=$messages, $link=$forcibly, $a, $debuginfo);
        return '';
    }

    $output = '';
    if (is_string($messages)){
        $messages = explode("\n", $messages);
    }
    foreach ($messages as $msg){
        $output .= '<p class="errormessage">' . s($msg) . '</p>';
    }
    $info = get_exception_info($ex);
    $output .= fatal_error($info->message, $info->moreinfourl, null, $info->backtrace, $info->debuginfo, $forcibly);
    return $output;
}

/**
 * Print hidden (if debugdeveloper is off) error in the moodle style from an exception
 *
 * @param \Exception|\Throwable   $ex
 * @param array|string  $messages
 * @param string        $not_debug_text
 *
 * @return string
 */
function print_hidden_error($ex, $messages=[], $not_debug_text=''){
    global $CFG;
    $error_msg = print_error($ex, $messages, true);
    if (!($CFG->debugdeveloper ?? 0)){
        $error_msg = $not_debug_text . \html_writer::div($error_msg, '', ['style' => 'display: none;']);
    }
    return $error_msg;
}

/**
 * Print error in the moodle style
 *
 * @param \Exception | \Throwable $exception
 *
 */
function cron_print_error($exception) {
    global $CFG;

    $obbuffer = '';
    $info = get_exception_info($exception);
    list($message, $moreinfourl, $link, $backtrace, $debuginfo, $errorcode) =
        [$info->message, $info->moreinfourl, $info->link, $info->backtrace, $info->debuginfo, $info->errorcode];

    $msgs = explode("\n", $message);
    foreach ($msgs as $msg){
        mtrace(s($msg));
    }
    if (empty($CFG->rolesactive)) {
        mtrace(get_string('installproblem', 'error'));
        //It is usually not possible to recover from errors triggered during installation, you may need to create a new database or use a different database prefix for new installation.
    }

    $labelsep = get_string('labelsep', 'langconfig');
    if (!empty($debuginfo)) {
        $debuginfo = s($debuginfo); // removes all nasty JS
        $label = get_string('debuginfo', 'debug') . $labelsep;
        mtrace($label.': '.$debuginfo);
    }
    if (!empty($backtrace)) {
        $label = get_string('stacktrace', 'debug') . $labelsep;
        mtrace($label.': '.format_backtrace($backtrace, true));
    }
    if ($obbuffer !== '' ) {
        $label = get_string('outputbuffer', 'debug') . $labelsep;
        mtrace($label.': '.s($obbuffer));
    }
}

/**
 * Check, does course categories (array of ids) has such course id
 * If none $courseid, return array of all course id from the current course categories
 *
 * @param array|string $course_cats - if string, id values should be delimiter by ','
 * @param              $courseid
 *
 * @return array|bool
 */
function course_cats_has_courseid($course_cats, $courseid=0){
    return SH::course_cats_has_courseid($course_cats, $courseid);
}
