<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * @package    block_ned_student_menu
 * @copyright  Michael Gardener <mgardener@cissq.com>
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

require_once($CFG->dirroot.'/mod/assignment/lib.php');

defined('MOODLE_INTERNAL') || die();

define('STUDENTMENU_ACTIVITYSETTING', '1');
define('STUDENTMENU_PASSGRADEPERCENT', '2');

use block_ned_student_menu as NED;

require_once($CFG->dirroot. '/blocks/ned_student_menu/common_lib.php');
require_once($CFG->dirroot. '/lib/grade/grade_item.php');

function block_ned_student_menu_assignment_status($mod, $userid) {
    global $CFG, $DB, $SESSION;

    if (isset($SESSION->completioncache)) {
        unset($SESSION->completioncache);
    }

    if ($mod->modname == 'assignment') {
        if (!($assignment = NED\get_module_db($mod->course, $mod->modname, $mod->instance))) {
            return false;
        }
        require_once($CFG->dirroot.'/mod/assignment/type/'.$assignment->assignmenttype.'/assignment.class.php');
        $assignmentclass = "assignment_$assignment->assignmenttype";
        $assignmentinstance = new $assignmentclass($mod->id, $assignment, $mod);

        if (!($submission = $assignmentinstance->get_submission($userid)) || empty($submission->timemodified)) {
            return false;
        }

        switch ($assignment->assignmenttype) {
            case "upload":
                if ($assignment->var4) { // If var4 enable then assignment can be saved.
                    if (!empty($submission->timemodified)
                        && (empty($submission->data2))
                        && (empty($submission->timemarked))) {
                        return 'saved';

                    } else if (!empty($submission->timemodified)
                        && ($submission->data2 = 'submitted')
                        && empty($submission->timemarked)) {
                        return 'submitted';
                    } else if (!empty($submission->timemodified)
                        && ($submission->data2 = 'submitted')
                        && ($submission->grade == -1)) {
                        return 'submitted';
                    }
                } else if (empty($submission->timemarked)) {
                    return 'submitted';
                }
                break;
            case "uploadsingle":
                if (empty($submission->timemarked)) {
                    return 'submitted';
                }
                break;
            case "online":
                if (empty($submission->timemarked)) {
                    return 'submitted';
                }
                break;
            case "offline":
                if (empty($submission->timemarked)) {
                    return 'submitted';
                }
                break;
        }
    } else if ($mod->modname == 'assign') {
        if (!($assignment = NED\get_module_db($mod->course, $mod->modname, $mod->instance))) {
            return false; // Doesn't exist.
        }

        if (!$submission = $DB->get_records('assign_submission',
            array('assignment' => $assignment->id, 'userid' => $userid), 'attemptnumber DESC', '*', 0, 1)
        ) {
            return false;
        } else {
            $submission = reset($submission);
        }

        $attemptnumber = $submission->attemptnumber;

        if (($submission->status == 'reopened') && ($submission->attemptnumber > 0)) {
            $attemptnumber = $submission->attemptnumber - 1;
        }

        if ($submissionisgraded = $DB->get_records('assign_grades',
            array('assignment' => $assignment->id, 'userid' => $userid,
                'attemptnumber' => $attemptnumber), 'attemptnumber DESC', '*', 0, 1
        )) {
            $submissionisgraded = reset($submissionisgraded);
            if ($submissionisgraded->grade > -1) {
                if (($submission->timemodified > $submissionisgraded->timemodified)
                    || ($submission->attemptnumber > $submissionisgraded->attemptnumber)) {
                    $graded = false;
                } else {
                    $graded = true;
                }
            } else {
                $graded = false;
            }
        } else {
            $graded = false;
        }

        if ($submission->status == 'draft') {
            if ($graded) {
                return 'submitted';
            } else {
                return 'saved';
            }
        }
        if ($submission->status == 'reopened') {
            return 'submitted';
        }
        if ($submission->status == 'submitted') {
            if ($graded) {
                return 'submitted';
            } else {
                return 'waitinggrade';
            }
        }
    } else {
        return false;
    }
}

/**
 * @param $completed
 * @param $incompleted
 * @param $saved
 * @param $notattempted
 * @param $waitingforgrade
 *
 * @return array[$items, $icons]
 */
function block_ned_student_get_activity_menu($completed, $incompleted, $saved, $notattempted, $waitingforgrade){
    $items = [];
    $icons = [];

    $link = function($number_activity, $key_activity){
        $text = $number_activity .' '. NED\str($key_activity, 'block_ned_student_menu');
        return html_writer::span($text, '');
    };

    $img = function($filename, $add_class='') {
        return NED\img($filename, 'my-progress-data icon ' . $add_class, NED\PLUGIN_NAME);
    };

    // Completed.
    if (!is_null($completed)) {
        $icons[] = $img('completed');
        if ($incompleted) {
            $items[] = $link($completed, 'completedsuccessful');
            // Incomplete.
            $items[] = $link($incompleted, 'completedunsuccessful');
            $icons[] = $img('incomplete');
        } else {
            $items[] = $link($completed, 'completed');
        }

    }

    // Draft.
    if (!is_null($saved) && $saved) {
        $items[] = $link($saved, 'draft');
        $icons[] = $img('saved');
    }

    // Not Attempted.
    if (!is_null($notattempted)) {
        $items[] = $link($notattempted, 'notattempted');
        $icons[] = $img('notattempted');
    }
    // Waiting for grade.
    if (!is_null($waitingforgrade)) {
        $items[] = $link($waitingforgrade, 'waitingforgrade');
        $icons[] = $img('unmarked');
    }

    return [$items, $icons];
}

function block_ned_student_pix_url($imagename, $component=null) {
    global $CFG, $OUTPUT;

    if ($CFG->version >= 2017051500) { // MDL 3.3+.
        return $OUTPUT->image_url($imagename, $component);
    } else {
        return $OUTPUT->pix_url($imagename, $component);
    }
}

/**
 * Return config for block on course page by course id
 * Don't use it in a course loop!
 *
 * @param        $course_id
 *
 * @param string $blockname
 *
 * @return bool|\stdClass
 * @throws \dml_exception
 */
function block_ned_student_menu_get_block_config($course_id, $blockname='block_ned_student_menu'){
    global $DB, $block_ned_student_menu_configs;

    if (!isset($block_ned_student_menu_configs)){
        $block_ned_student_menu_configs = [];
    }
    if (!isset($block_ned_student_menu_configs[$course_id][$blockname])){
        $context = context_course::instance($course_id);
        $blockdata = $DB->get_record('block_instances', ['blockname' => $blockname, 'parentcontextid' => $context->id]);
        if (!$blockdata){
            return false;
        }
        $config = unserialize(base64_decode($blockdata->configdata));
        $config->block_id = $blockdata->id;
        $config->course_id = $course_id;
        $block_ned_student_menu_configs[$course_id][$blockname] = $config;
    }

    return $block_ned_student_menu_configs[$course_id][$blockname];
}

/**
 * Save config for block, return true if success
 *
 * @param        $course_id
 * @param        $config
 * @param string $blockname
 *
 * @return bool
 * @throws \dml_exception
 */
function block_ned_student_menu_set_block_config($course_id, $config, $blockname='block_ned_student_menu'){
    global $DB, $block_ned_student_menu_configs;

    if (!isset($block_ned_student_menu_configs)){
        $block_ned_student_menu_configs = [];
    }

    $configdata = base64_encode(serialize($config));
    $context = context_course::instance($course_id);
    if ($DB->set_field('block_instances', 'configdata', $configdata,
        ['blockname' => $blockname, 'parentcontextid' => $context->id])){
        $block_ned_student_menu_configs[$course_id][$blockname] = $config;
        return true;
    }
    return false;
}

/**
 * Get block_ned_teacher_tools config for course, based on site block_ned_teacher_tools config and course block settings
 * @param $course_id
 *
 * @return object
 * @throws \dml_exception
 */
function block_ned_student_menu_get_teacher_tools_block_config($course_id){
    $site_config = get_config('block_ned_teacher_tools');
    $block_config = block_ned_student_menu_get_block_config($course_id, $blockname='ned_teacher_tools');
    $config = (object)array_merge(['resubmission_assignments' => []], (array)$site_config, (array)$block_config);
    $config->resubmission_assignments = $config->enableresubmissions ? $config->resubmission_assignments : [];
    return $config;
}
